/*-----------------------------------------------------------------------------
 * Copyright (c) 1991,1992 Southeastern Universities Research Association,
 *                         Continuous Electron Beam Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
 * Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
 *-----------------------------------------------------------------------------
 * 
 * Description:
 *	 dictionary tree structure utilities
 *	
 * Author:  Jie Chen
 *          CEBAF Data Acquisition Group
 *
 * Revision History:
 *   $Log: cefDictTree.c,v $
 *   Revision 1.1.1.1  1996/09/24 14:21:27  chen
 *   Initial import to coda_2.0
 *
*	  Revision 1.1  1995/03/17  15:00:06  chen
*	  Initial revision
*
 *
 */
#include <stdio.h>
#include <string.h>
#include "cefDictTree.h"

extern char *strsave ();

/***************************************************************************
 *           void tree_d_print (TREE_NODE *root)                           *
 * Description:                                                            *
 *     Debug tree printing utility                                         *
 **************************************************************************/
void tree_d_print(root)
TREE_NODE *root;
{
  TREE_NODE *p;

  p = root;
   
  if(p!=NULL){
    if(p->parent != NULL)
      printf("node is %s  and its parent is %s\n",p->name,p->parent->name);
    else
      printf("node is %s \n",p->name);
    tree_d_print(p->mleft);
    tree_d_print(p->rsibling);
  }
}

/***************************************************************************
 *           void print_forest (FOREST *fr)                                *
 * Description:                                                            *
 *     Debug forest printing utility                                       *
 **************************************************************************/
void print_forest (fr)
FOREST *fr;
{
  int i;

  for (i = 0; i < fr->numTrees; i++){
    printf("tree number %d\n",i);
    tree_d_print (fr->root[i]);
  }
}

/***************************************************************************
 *       void updateGlbTreeNode (FOREST *fr)                               *
 * Description:                                                            *
 *     Update global dictionary tree structure according to the            *
 *     FOREST structure. Only called from parser side                      *
 **************************************************************************/
void updateGlbTreeNode (fr)
FOREST *fr;
{
  int i = 0;

#ifdef _CODA_DEBUG
  printf("num of trees is %d\n",fr->numTrees);
#endif
  for (i = 0; i < fr->numTrees; i++){
    root_d[i] = fr->root[fr->numTrees - i - 1];
    /* top level has no rifht sibling */
    root_d[i]->rsibling = 0;
  }
  num_of_tree = fr->numTrees;
#ifdef _CODA_DEBUG
  for (i = 0; i < num_of_tree; i++){
    printf("trees number %d\n", i);
    tree_d_print (root_d[i]);
  }
#endif
}

/***************************************************************************
 *       void newTreeNode (void)                                           *
 * Description:                                                            *
 *      Create a new tree node                                             *
 **************************************************************************/
TREE_NODE *newTreeNode ()
{
  TREE_NODE *p = (TREE_NODE *)malloc (sizeof(TREE_NODE));
  if (p == NULL){
    fprintf(stderr, "Fatal error: Cannot allocate memory \n");
    exit (1);
  }
  p->tag_v = 0;
  p->name = 0;
  p->title = 0;
  p->mleft = 0;
  p->rsibling = 0;
  p->parent = 0;
  return p;
}

/**************************************************************************
 *       void updateTreeNode (node, tag_v, name, title)                   *
 * Description:                                                           *
 *       Filling the element of tree structure pointed by node            *
 **************************************************************************/
void updateTreeNode (node, tag_v, name, title)
TREE_NODE *node;
char      *tag_v;
char      *name;
char      *title;
{
  sscanf (tag_v, "%x", &(node->tag_v));
  node->name = strsave (name);
  node->title = strsave (title);
}

/**************************************************************************
 *       void deleteDictTree (TREENODE *dr)                               *
 * Description:                                                           *
 *       Delete the whole tree pointed by 'dr'                            *
 **************************************************************************/
void deleteDictTree (dr)
TREE_NODE *dr;
{
  TREE_NODE *mleft, *rsibling;

  mleft = dr->mleft;
  rsibling = dr->rsibling;
  if (dr->name != NULL)
    free (dr->name);
  if (dr->title != NULL)
    free (dr->title);
  free (dr);
  if(mleft != NULL)
    deleteDictTree (mleft);
  if(rsibling != NULL)
    deleteDictTree (rsibling);
}

/**************************************************************************
 *       void mleftOfTree (tree, forest)                                  *
 * Description:                                                           *
 *       Make the 'tree' as the most left child of the forest 'forest'    *
 **************************************************************************/
void mleftOfTree (tree, forest)
TREE_NODE *tree;
FOREST    *forest;
{
  tree->mleft = forest->root[forest->numTrees - 1];
}

/**************************************************************************
 *       void parentOfForest (tree, forest)                               *
 * Description:                                                           *
 *       Make the 'tree' as the parent the forest 'forest'                *
 **************************************************************************/
void parentOfForest (fr, tree)
FOREST *fr;
TREE_NODE *tree;
{
  int i = 0;

  for (i = 0; i < fr->numTrees; i++){
    fr->root[i]->parent = tree;
  }
}

/**************************************************************************
 *       FOREST* newForest (void)                                         *
 * Description:                                                           *
 *     Create a new forest                                                *
 **************************************************************************/
FOREST *newForest ()
{
  FOREST *fr;

  fr = (FOREST *)malloc (sizeof(FOREST));
  if (fr == NULL){
    fprintf(stderr,"Cannot allocate memory for forest\n");
    exit (1);
  }
  fr->numTrees = 0;
  return fr;
}

/**************************************************************************
 *       void addTreeToForest (fr, tree)                                  *
 * Description:                                                           *
 *     Add a single tree node to the forest                               *
 **************************************************************************/
void addTreeToForest (fr, tree)
FOREST *fr;
TREE_NODE *tree;
{
  if (fr->numTrees == 0){
    fr->root[fr->numTrees] = tree;
    fr->numTrees += 1;
  }
  else{
    tree->rsibling = fr->root[fr->numTrees - 1];
    fr->root[fr->numTrees] = tree;
    fr->numTrees += 1;
  }
}

/***************************************************************************
 *       void deleteForest (FOREST *fr)                                    *
 * Description:                                                            *
 *     Release memory of forest without deleting those tree nodes inside it*
 **************************************************************************/
void deleteForest (fr)
FOREST *fr;
{
  fr->numTrees = 0;
  free (fr);
}
