/*
 *      Original Author: Ben-chin Cha
 *      Date:            8-10-92
 *
 *      Experimental Physics and Industrial Control System (EPICS)
 *
 *      Copyright 1991, the Regents of the University of California,
 *      and the University of Chicago Board of Governors.
 *
 *      This software was produced under  U.S. Government contracts:
 *      (W-7405-ENG-36) at the Los Alamos National Laboratory,
 *      and (W-31-109-ENG-38) at Argonne National Laboratory.
 *
 *      Initial development by:
 *              The Controls and Automation Group (AT-8)
 *              Ground Test Accelerator
 *              Accelerator Technology Division
 *              Los Alamos National Laboratory
 *
 *      Co-developed with
 *              The Controls and Computing Group
 *              Accelerator Systems Division
 *              Advanced Photon Source
 *              Argonne National Laboratory
 *
 * Modification Log:
 * -----------------
 * .01  08-11-92        bkc     Using caGlobals structure, all the global 
 * 				functions begins with ca_ . 
 * .02  09-18-92        bkc     Create ca_populate_info_array function
 * .03  01-29-93        bkc     Fix bug in put float value to IOC 
 * .04  08-31-93        bkc     Add if condition for usleep
 * .04  09-07-93        bkc     Set Waveform size to max 16000 byte
 * .05  09-20-93        bkc     Fix the ca_populate_info_array for enum types
 * .06  mm-dd-yy        iii     Comment
 */

#include "chandata.h"
#include "tcl.h"

struct caGlobals CA = {
	0,	/* CA_ERR */
	0,	/* devprflag */
	0.3, 	/* PEND_IO_TIME */
	30.0, 	/* PEND_IOLIST_TIME */
	0.01}; 	/* PEND_EVENT_TIME */

extern chandata *pchandata;

int dbr_type = DBR_STS_FLOAT;

double atof();

union val_tag {
        char cv[16000];
        char pv[400][40];
        short sv[8000];
        int iv[4000];
        float fv[4000];
        double dv[2000];
        } wf_buf;

void ca_get_dbr_sts_double_callback();
void ca_get_dbr_sts_string_callback();
void ca_get_dbr_sts_string_callback3();
void ca_get_dbr_ctrl_double_callback();

int ca_get_dbr_string();
int ca_get_dbr_float();
int ca_get_dbr_sts_string();
int ca_get_dbr_sts_float();
int ca_get_dbr_sts_double();
int ca_get_dbr_gr_enum();
int ca_get_dbr_ctrl_double();

int ca_put_dbr_float();
int ca_put_dbr_double();
int ca_put_dbr_string();

void ca_sleep_time();
void ca_check_command_error(); 
void ca_check_array_return_code();
void ca_check_return_code();
void ca_execerror();

void ca_find_dev();
int ca_get_conn_data();
int ca_put_conn_data();
void ca_put_real_value();
double ca_get_real_value2();
double ca_get_real_value();
char *ca_get_string3();
char *ca_get_string2();
char *ca_get_string();

double *ca_get_wave_form();
int ca_put_wave_form();
void ca_get_all();
int ca_search_list();
int ca_pvlist_search();
void ca_get_string_array();
void ca_get_info_array();
void ca_populate_info_array();
void ca_get_error_array();
void ca_old_status_array();
void ca_get_status_array();
void ca_get_double_array2();
void ca_get_all_double_array();
void ca_get_double_array();
void ca_put_string_array();
void ca_put_double_array();

Tcl_Interp *pinterp;

/****************************************************
   sleep for micro seconds
*****************************************************/
void ca_sleep_time(t)
double t;    /* wait in seconds can be decimal*/
{
/* usleep isn't ANSI-C/POSIX
*/
unsigned u;
#if TARGET_CPU == HP_UX
	sleep((unsigned int)t);
#else
	u = 1000000 * t;
	usleep(u);
#endif

}

/****************************************************
   check for command return error code 
*****************************************************/
void ca_check_command_error(i) 
int i;
{
	if (i != 0) 
		CA.CA_ERR = CA_FAIL;
		else CA.CA_ERR = CA_SUCCESS;

}

/**************************************************
 * check a list of array error code return by CA
 **************************************************/
void ca_check_array_return_code(status)
int status;
{
  if (status == ECA_NORMAL) {
		CA.CA_ERR = CA_SUCCESS;
		return;
		}
    else CA.CA_ERR = CA_FAIL;

	
  if (status == ECA_TIMEOUT)
        ca_execerror("The operation timed out: "," on list of devices.");
} 

/**************************************************
 * check error code return by CA for a single device
 **************************************************/
void ca_check_return_code(status)
int status;
{
  if (status == ECA_NORMAL) {
		CA.CA_ERR = CA_SUCCESS;
		pchandata->error = CA_SUCCESS;
		return;
		}
    else { 
	CA.CA_ERR = CA_FAIL;
	pchandata->error = -2;
	}
	
  if (status == ECA_TIMEOUT)
        ca_execerror("The operation timed out: ",ca_name(pchandata->chid));
  if (status == ECA_GETFAIL)
        ca_execerror("A local database get failed: ",ca_name(pchandata->chid));
  if (status == ECA_BADCHID)
        ca_execerror("Unconnected or corrupted chid: ",ca_name(pchandata->chid));
  if (status == ECA_BADCOUNT)
        ca_execerror("More than native count requested: ",ca_name(pchandata->chid));
  if (status == ECA_BADTYPE)
        ca_execerror("Unknown GET_TYPE: ",ca_name(pchandata->chid));
  if (status == ECA_STRTOBIG)
        ca_execerror("Unusually large string supplied: ",ca_name(pchandata->chid));
  if (status == ECA_ALLOCMEM)
        ca_execerror("Unable to allocate memory: ",ca_name(pchandata->chid));
}

/**************************************************
 *   print error message
 *************************************************/
void ca_execerror(s,t)
char *s, *t;
{

fprintf(stderr,"%s %s\n ",s,t);
}

void ca_execerrorp(s,t)
char *s, *t;
{

/* fprintf(stderr,"%s %s\n ",s,t); */
  Tcl_AppendResult(pinterp, s, t, (char *) NULL);
}

/***************************************************
 * put real value to ioc channel
 **************************************************/
void ca_put_real_value(name,value)
char *name;
double value;
{
int status;
	ca_find_dev(name,pchandata);
	if (pchandata->type != TYPENOTCONN) {
	status = ca_put_conn_data(DBR_DOUBLE,pchandata,&value);
	if (status == ECA_NORMAL)
	value = pchandata->value;
	}
}

/**************************************************
 *  return a real value for a given channel name 
 **************************************************/
double ca_get_real_value2(name)
char *name;
{
double value=0;
	ca_find_dev(name,pchandata);
	if (pchandata->type != TYPENOTCONN) {
	if (pchandata->type != DBR_STRING) 
		ca_get_conn_data(DBR_STS_DOUBLE,pchandata);
	else
		ca_get_conn_data(DBR_STS_STRING,pchandata);

	value = pchandata->value;
	}
	return value;
}

/**************************************************
 *  return return value , status, severity 
 **************************************************/
void ca_get_all(name,vals)
char *name;
double *vals;
{
double value;
int status;
        ca_find_dev(name,pchandata);
        status = ca_get_conn_data(DBR_STS_DOUBLE,pchandata);
        if (status == ECA_NORMAL) {
        value = pchandata->value;
        *vals = value;
        value = pchandata->status;
        *(vals+1) = value;
        value = pchandata->severity;
        *(vals+2) = value;

	if (CA.devprflag > 0) {
       	 	fprintf(stderr,"name=%s,",name);
       		fprintf(stderr,"value=%f, status=%f, severity=%f\n ",
               	         *vals,*(vals+1),*(vals+2));
       	 	}

        }
}

/**************************************************
 *  return a real value for a given channel name by using callback 
 **************************************************/
double ca_get_real_value(name)
char *name;
{
int status;
int count=0,max;
double value=0;

	max = 10. / CA.PEND_EVENT_TIME;

	ca_find_dev(name,pchandata);
	if (pchandata->type != TYPENOTCONN) {
		ca_puser(pchandata->chid) = pchandata;
		pchandata->error = CA_WAIT;
             	status = ca_get_callback(DBR_STS_DOUBLE,pchandata->chid,
				ca_get_dbr_sts_double_callback,pchandata);
		if (status != ECA_NORMAL) {
                ca_execerror("ca_get_callback failed on",ca_name(pchandata->chid));
			ca_check_return_code(status);
			}
		else {
                ca_pend_event(CA.PEND_EVENT_TIME);

		/* wait for ca_get_callback to finish */

		while(pchandata->error == CA_WAIT) {
			count++;
			if (count > max) break;
			ca_pend_event(CA.PEND_EVENT_TIME); 
			}	

		value = pchandata->value;
		}
	}
	return value;
}

/**************************************************
 *  return string value back for a given channel name 
 **************************************************/
char *ca_get_string2(name)
char *name;
{
int status,count=0,max;
char *string;
	ca_find_dev(name,pchandata);
	if (pchandata->type != TYPENOTCONN) 
	ca_get_conn_data(DBR_STS_STRING,pchandata);

	string = pchandata->string;
	return(string);
}

/**************************************************
 *  return string value back for a given channel name 
 **************************************************/
char *ca_get_string(name)
char *name;
{
  int status,count=0,max;
  char *string;
  
  ca_find_dev(name,pchandata);
  
  if (pchandata->type != TYPENOTCONN) {
    ca_puser(pchandata->chid) = pchandata;
    pchandata->error = CA_WAIT;
    status = ca_get_callback(DBR_STS_STRING,pchandata->chid,
			     ca_get_dbr_sts_string_callback,pchandata);
    
    if (status != ECA_NORMAL) {
    ca_execerror("ca_get_callback failed on",ca_name(pchandata->chid));
    ca_check_return_code(status);
  }
    else {
      ca_pend_io(CA.PEND_IO_TIME);
      
      /* wait for ca_get_callback to finish */
      
      max = 10. / CA.PEND_EVENT_TIME;
      
      while(pchandata->error == CA_WAIT) {
	count++;
	if (count > max) break;
	ca_pend_event(CA.PEND_EVENT_TIME); 
      }	
    }
  }
  
  string = pchandata->string;
  return string;
}

char *ca_get_string3(name)
char *name;
{
  int status,count=0,max;
  char *string;
  
  ca_find_dev(name,pchandata);
  
  if (pchandata->type != TYPENOTCONN) {
    ca_puser(pchandata->chid) = pchandata;
    pchandata->error = CA_WAIT;
    status = ca_get_callback(DBR_STS_STRING,pchandata->chid,
			     ca_get_dbr_sts_string_callback3,pchandata);
    
    if (status != ECA_NORMAL) {
      ca_execerror("ca_get_callback failed on",ca_name(pchandata->chid));
      ca_check_return_code(status);
    }
    else {
      ca_pend_io(CA.PEND_IO_TIME);
      ca_pend_event(CA.PEND_EVENT_TIME);
    }
  }
}

/**************************************************
 * find the channel name from IOC 
 **************************************************/
void ca_find_dev(name,pchandata)
char *name;
chandata *pchandata;
{
  int status;
  
  /* populate hash table here return the address */
  
  pchandata = (chandata *)ca_check_hash_table(name);
  
  status = ca_pend_io(CA.PEND_IO_TIME);
  ca_check_return_code(status);
  pchandata->type = ca_field_type(pchandata->chid);
  if (pchandata->state != cs_closed)
    pchandata->state = ca_state(pchandata->chid);
  else 
    CA.CA_ERR = CA_FAIL;
  if (pchandata->type == TYPENOTCONN) {
    ca_execerror(ca_name(pchandata->chid),"--- Invalid device name");
    /*  clear if never conn */
    if (pchandata->state == cs_never_conn) 
      status = ca_clear_channel(pchandata->chid);
    if (status != ECA_NORMAL)   ca_check_return_code(status);
    pchandata->state = ca_state(pchandata->chid);
    ca_field_type(pchandata->chid) == TYPENOTCONN;
  }
  if(pchandata->type > 0 && pchandata->type != 3) 
    pchandata->form = 0;  /* not enum type */ 
}

void ca_find_devp(interp,name,pchandata)
char *name;
chandata *pchandata;
{
  int status;
  pinterp = interp;
  /* populate hash table here return the address */
  
  pchandata = (chandata *)ca_check_hash_table(name);
  
  status = ca_pend_io(CA.PEND_IO_TIME);
  ca_check_return_code(status);
  pchandata->type = ca_field_type(pchandata->chid);
  if (pchandata->state != cs_closed)
    pchandata->state = ca_state(pchandata->chid);
  else 
    CA.CA_ERR = CA_FAIL;
  if (pchandata->type == TYPENOTCONN) {
    ca_execerrorp(ca_name(pchandata->chid),"--- Invalid device name");
    /*  clear if never conn */
    if (pchandata->state == cs_never_conn) 
      status = ca_clear_channel(pchandata->chid);
    if (status != ECA_NORMAL)   ca_check_return_code(status);
    pchandata->state = ca_state(pchandata->chid);
    ca_field_type(pchandata->chid) == TYPENOTCONN;
  }
  if(pchandata->type > 0 && pchandata->type != 3) 
    pchandata->form = 0;  /* not enum type */ 
}

/**************************************************
 * get channel data back according to db request type
 **************************************************/
int ca_get_conn_data(dbr_type,pchandata)
int dbr_type;
chandata *pchandata;
{
int status;
float value=0.;

switch (dbr_type) {

        case DBR_STRING:
                status = ca_get_dbr_string(pchandata);
                break;

        case DBR_FLOAT:
                status = ca_get_dbr_float(pchandata);
                break;

        case DBR_STS_STRING:
                status = ca_get_dbr_sts_string(pchandata);
                break;

        case DBR_STS_FLOAT:
                status = ca_get_dbr_sts_float(pchandata);
                break;

        case DBR_STS_DOUBLE:
                status = ca_get_dbr_sts_double(pchandata);
                break;

        case DBR_GR_ENUM:
                status = ca_get_dbr_gr_enum(pchandata);
                break;

        case DBR_CTRL_DOUBLE:
                status = ca_get_dbr_ctrl_double(pchandata);
                break;

        }
return status;
}

/**************************************************
 * get channel value for DBR_STRING
**************************************************/
int ca_get_dbr_string(pchandata)
chandata *pchandata;
{
int status;
char value[MAX_STRING_SIZE];

  if (pchandata->type != TYPENOTCONN) {

        status = ca_get(DBR_STRING,pchandata->chid,value);
        if (status != ECA_NORMAL) {
                ca_execerror("ca_get failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
        	status = ca_pend_io(CA.PEND_IO_TIME);
	        ca_check_return_code(status);
		if (status == ECA_NORMAL)
	        strcpy(pchandata->string,value);
		}
          return status;
        }
        else return ECA_BADCHID;
}

/**************************************************
 * get channel value for DBR_FLOAT
**************************************************/
int ca_get_dbr_float(pchandata)
chandata *pchandata;
{
int status;
float value=0.;

  if (pchandata->type != TYPENOTCONN) {

        status = ca_get(DBR_FLOAT,pchandata->chid,&value);
        if (status != ECA_NORMAL) {
                ca_execerror("ca_get failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
	        status = ca_pend_io(CA.PEND_IO_TIME);
	        ca_check_return_code(status);
		if (status == ECA_NORMAL)
	        pchandata->value = value;
		}
          return status;
        }
        else return ECA_BADCHID;
}

/**************************************************
 * get channel value for DBR_STS_STRING
**************************************************/
int ca_get_dbr_sts_string(pchandata)
chandata *pchandata;
{
int status;
struct dbr_sts_string dbr;

  if (pchandata->type != TYPENOTCONN) {
        status = ca_get(DBR_STS_STRING,pchandata->chid,&dbr);
        if (status != ECA_NORMAL) {
                ca_execerror("ca_get failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
          status = ca_pend_io(CA.PEND_IO_TIME);
          ca_check_return_code(status);

	if (status == ECA_NORMAL) {
		if (CA.devprflag > 1) {
		        fprintf(stderr,"\tvalue=%s\n",dbr.value);
		        fprintf(stderr,"\tstatus=%d\n",dbr.status);
		        fprintf(stderr,"\tseverity=%d\n",dbr.severity);
			}
	        pchandata->status = dbr.status;
	        pchandata->severity = dbr.severity;
		strcpy(pchandata->string,dbr.value);
	        pchandata->value = atof(dbr.value);
			}
		}
          return status;
        }
        else return ECA_BADCHID;

}

/**************************************************
 * get channel value for DBR_STS_FLOAT
**************************************************/
int ca_get_dbr_sts_float(pchandata)
chandata *pchandata;
{
int status;
struct dbr_sts_float dbr;

  if (pchandata->type != TYPENOTCONN) {
        status = ca_get(DBR_STS_FLOAT,pchandata->chid,&dbr);
        if (status != ECA_NORMAL) {
                ca_execerror("ca_get failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
          status = ca_pend_io(CA.PEND_IO_TIME);
          ca_check_return_code(status);
	if (status == ECA_NORMAL) {
	if (CA.devprflag > 1) {
       		 fprintf(stderr,"\tvalue=%f\n",dbr.value);
		 fprintf(stderr,"\tstatus=%d\n",dbr.status);
		 fprintf(stderr,"\tseverity=%d\n",dbr.severity);
		}
	        pchandata->value = dbr.value;
	        pchandata->status = dbr.status;
	        pchandata->severity = dbr.severity;
			}
		}
          return status;
        }
        else return ECA_BADCHID;

}

/**************************************************
 * get channel value for DBR_STS_DOUBLE
**************************************************/
int ca_get_dbr_sts_double(pchandata)
chandata *pchandata;
{
int status;
struct dbr_sts_double dbr;


  if (pchandata->type != TYPENOTCONN) {
        status = ca_get(DBR_STS_DOUBLE,pchandata->chid,&dbr);
        if (status != ECA_NORMAL) {
                ca_execerror("ca_get failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
          status = ca_pend_io(CA.PEND_IO_TIME);
          ca_check_return_code(status);
	if (status == ECA_NORMAL) {
	if (CA.devprflag > 1) {
       		fprintf(stderr,"\tvalue=%f\n",dbr.value);
	        fprintf(stderr,"\tstatus=%d\n",dbr.status);
	        fprintf(stderr,"\tseverity=%d\n",dbr.severity);
			}
       		pchandata->value = dbr.value;
	        pchandata->status = dbr.status;
	        pchandata->severity = dbr.severity;
		}
	}
          return status;
        }
        else return ECA_BADCHID;

}


/**************************************************
 * get channel status string for DBR_GR_ENUM  (name.STAT)
**************************************************/
int ca_get_dbr_gr_enum(pchandata)
chandata *pchandata;
{
int status,i;
struct dbr_gr_enum dbr;

  if (pchandata->type != TYPENOTCONN) {

        status = ca_get(DBR_GR_ENUM,pchandata->chid,&dbr);
        if (status != ECA_NORMAL) {
                ca_execerror("ca_get failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
	        status = ca_pend_io(CA.PEND_IO_TIME);
	        ca_check_return_code(status);
		if (status == ECA_NORMAL)
	        pchandata->value = dbr.value;
	        pchandata->status= dbr.status;
	        pchandata->severity = dbr.severity;
		strcpy(pchandata->string, dbr.strs[dbr.value]);

if (CA.devprflag > 0) 
	fprintf(stderr,"name=%s,status=%d,severity=%d,value=%d\n",
	ca_name(pchandata->chid),dbr.status,dbr.severity,dbr.value);

	/* print out status string defined in database */

if (CA.devprflag > 2) 
	for (i=0;i<dbr.no_str;i++) 
	fprintf(stderr,"%d      %s\n",i,dbr.strs[i]);
	

		}
          return status;
        }
        else return ECA_BADCHID;
}

/**************************************************
 * get channel value for DBR_CTRL_DOUBLE
**************************************************/
int ca_get_dbr_ctrl_double(pchandata)
chandata *pchandata;
{
int status;
float value=0.;
struct dbr_ctrl_double dbr;

  if (pchandata->type != TYPENOTCONN) { 

        status = ca_get(DBR_CTRL_DOUBLE,pchandata->chid,&dbr);
        if (status != ECA_NORMAL) {
                ca_execerror("ca_get failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
          status = ca_pend_io(CA.PEND_IO_TIME);
          ca_check_return_code(status);
	if (status == ECA_NORMAL) {
if (CA.devprflag > 1) {
        fprintf(stderr,"\tprecision=%d\n",dbr.precision);
        fprintf(stderr,"\tRISC_pad0=%d\n",dbr.RISC_pad0);
        fprintf(stderr,"\tunits=%s\n",dbr.units);
	fprintf(stderr,"\tdisp_limit : [ %f : %f ]\n",
		dbr.lower_disp_limit, dbr.upper_disp_limit);
	fprintf(stderr,"\tctrl_limit : [ %f : %f ]\n",
		dbr.lower_ctrl_limit, dbr.upper_ctrl_limit);
        fprintf(stderr,"\tupper_alarm_limit=%f\n",dbr.upper_alarm_limit);
        fprintf(stderr,"\tupper_warning_limit=%f\n",dbr.upper_warning_limit);
        fprintf(stderr,"\tlower_warning_limit=%f\n",dbr.lower_warning_limit);
        fprintf(stderr,"\tlower_alarm_limit=%f\n",dbr.lower_alarm_limit);
        fprintf(stderr,"\tvalue=%f\n",dbr.value);
        fprintf(stderr,"\tstatus=%d\n",dbr.status);
        fprintf(stderr,"\tseverity=%d\n",dbr.severity);
	}

        pchandata->value = dbr.value;
        pchandata->uopr = dbr.upper_disp_limit;
        pchandata->lopr = dbr.lower_disp_limit;
        pchandata->upper_alarm_limit = dbr.upper_alarm_limit;
        pchandata->upper_warning_limit = dbr.upper_warning_limit ;
        pchandata->lower_warning_limit = dbr.lower_warning_limit ;
        pchandata->lower_alarm_limit = dbr.lower_alarm_limit ;
        pchandata->upper_ctrl_limit = dbr.upper_ctrl_limit ;
        pchandata->lower_ctrl_limit = dbr.lower_ctrl_limit ;
        pchandata->status = dbr.status;
        pchandata->severity = dbr.severity;
        strcpy(pchandata->units, dbr.units);
        if (pchandata->max < dbr.value) pchandata->max = dbr.value;
        if (pchandata->min > dbr.value) pchandata->min = dbr.value;
	}
	}
          return status;
        }
        else return ECA_BADCHID;

}

/********************************************************
 * write value to the connected channel according to db request type
 ********************************************************/
int ca_put_conn_data(dbr_type,pchandata,pvalue)
int dbr_type;
chandata *pchandata;
void *pvalue;
{
int status;

switch (dbr_type) {

        case DBR_FLOAT:
                status = ca_put_dbr_float(pchandata,pvalue);
                break;

        case DBR_DOUBLE:
                status = ca_put_dbr_double(pchandata,pvalue);
                break;

        case DBR_STRING:
                status = ca_put_dbr_string(pchandata,pvalue);
                break;
        }
return status;
}

/********************************************************
 * write a real value to the connected channel 
 ********************************************************/
int ca_put_dbr_float(pchandata,pvalue)
chandata *pchandata;
void *pvalue;
{
int status;
float value;

  value = *(float *)pvalue;

  pchandata->type = ca_field_type(pchandata->chid);

  if (pchandata->type !=TYPENOTCONN) {
        status = ca_put(DBR_FLOAT,pchandata->chid,pvalue);
	if (status != ECA_NORMAL) {
                ca_execerror("ca_put failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
         	status = ca_pend_io(CA.PEND_IO_TIME);
	        ca_check_return_code(status);
		if (status == ECA_NORMAL) 
	        pchandata->setpoint = value;
		}
         return status;
        }
        else return ECA_BADCHID;
}

/********************************************************
 * write a double value to the connected channel 
 ********************************************************/
int ca_put_dbr_double(pchandata,pvalue)
chandata *pchandata;
void *pvalue;
{
int status;
double value;

  value = *(double *)pvalue;

  pchandata->type = ca_field_type(pchandata->chid);

  if (pchandata->type != TYPENOTCONN) {
        status = ca_put(DBR_DOUBLE,pchandata->chid,pvalue);
 	if (status != ECA_NORMAL) {
                ca_execerror("ca_put failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
  	else {
         	status = ca_pend_io(CA.PEND_IO_TIME);
         	ca_check_return_code(status);
		if (status == ECA_NORMAL) 
	        pchandata->setpoint = value;
		}
         return status;
        }
        else return ECA_BADCHID;
}

/********************************************************
 * write a string value to the connected channel 
 ********************************************************/
int ca_put_dbr_string(pchandata,pvalue)
chandata *pchandata;
void *pvalue;
{
int status;

  pchandata->type = ca_field_type(pchandata->chid);

  if (pchandata->type != TYPENOTCONN) {
        status = ca_put(DBR_STRING,pchandata->chid,pvalue);
	if (status != ECA_NORMAL) {
                ca_execerror("ca_put failed on",ca_name(pchandata->chid));
                ca_check_return_code(status);
                }
	else {
       	  	status = ca_pend_io(CA.PEND_IO_TIME);
         	ca_check_return_code(status);
		if (status == ECA_NORMAL) 
         	pchandata->setpoint = atof(pchandata->string); 
		}
         return status;
        }
        else return ECA_BADCHID;
}


/**************************************************
 *  casearch for a device array
 *  time out return CA_FAIL else return 0
 *************************************************/
ca_search_list(noName,pvName)
int noName;
char **pvName;
{
int command_error=0;
chandata *list;

        command_error = ca_pvlist_search(noName,pvName,&list);
	return(command_error);
}


/**************************************************
 *  casearch for a device array
 *  time out return CA_FAIL else return 0
 *************************************************/
int ca_pvlist_search(noName,pvName,list)
int noName;
char **pvName;
chandata **list;
{
int i,status,command_error=0;
chandata *ptr,*pnow,*phead;

	phead = (chandata *)list;
	pnow = phead;
	for (i=0;i<noName;i++) {
		pchandata = (chandata *)ca_check_hash_table(pvName[i]);
		pnow->next = pchandata;
		pchandata->next = NULL;
		pnow = pchandata;
/*fprintf(stderr,"%s, %x, next=%x\n",ca_name(pchandata->chid),pchandata,pchandata->next);
 */
		}
        status = ca_pend_io(CA.PEND_IOLIST_TIME);
        if (status != ECA_NORMAL) ca_check_array_return_code(status);

	pnow = phead->next; i=0;
	while (pnow)  {
		pchandata = pnow;
/*fprintf(stderr,"%s, %x, next=%x\n",ca_name(pchandata->chid),pchandata,pchandata->next);
 */
		if (pchandata->error == CA_WAIT) status = ECA_TIMEOUT;
		pchandata->type = ca_field_type(pchandata->chid);
		if (pchandata->state != cs_closed)
			 pchandata->state = ca_state(pchandata->chid);

		if (pchandata->type == TYPENOTCONN) {
			fprintf(stderr,"%-30s  ***  device not found\n",pvName[i]);
			pchandata->error = CA_WAIT;
			command_error = CA_FAIL;

/*  clear channel if never conn */

if (pchandata->state == cs_never_conn) {
	if (status != ECA_NORMAL)   ca_check_return_code(status);
	 ca_clear_channel(pchandata->chid);
	pchandata->state = ca_state(pchandata->chid);
	}

			}
		pnow = pnow->next; i++;
		}

	ca_check_command_error(command_error);
	if (status == ECA_NORMAL && command_error == CA_SUCCESS) return(CA_SUCCESS);
		else return(CA_FAIL);
}

/**************************************************
 *  get string value back for a device array
 *************************************************/
void ca_get_string_array(noName,pvName,value)
int noName;
char **pvName;
char **value;
{
int i,status,command_error=0;
int count=0,max;
chandata *list;
chandata *snode;

	command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list;
        while (snode)  {
                pchandata = snode;
		if (pchandata->type != TYPENOTCONN) {
                        ca_puser(pchandata->chid) = pchandata;
                        pchandata->error = CA_WAIT;
                        status = ca_get_callback(DBR_STS_STRING,pchandata->chid,
                                ca_get_dbr_sts_string_callback,pchandata);

                        if (status != ECA_NORMAL) {
                                 command_error = CA_FAIL;
                                 ca_check_array_return_code(status);
                                }
			}


                snode = snode->next;
                }
	
        max = 10. / CA.PEND_EVENT_TIME;

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {

                        /* wait for ca_get_callback to finish */

                        while (pchandata->error == CA_WAIT) {
                                count++;
                                if (count > max) break;
                                ca_pend_event(CA.PEND_EVENT_TIME);
                                }
               		 *(value+i) = pchandata->string;
			}
                snode = snode->next; i++;
		}	

	ca_check_command_error(command_error);
}
 


/**************************************************
 *  list information for a device array
 *************************************************/
void ca_get_info_array(noName,pvName,value)
int noName;
char **pvName;
double  *value;
{
int i,status,command_error=0;
int count=0,max;
chandata *list;
chandata *snode;


	command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list;
        while (snode)  {
                pchandata = snode;
		if (pchandata->type != TYPENOTCONN) {
                ca_puser(pchandata->chid) = pchandata;
                pchandata->error = CA_WAIT;

		if (pchandata->type == DBR_STRING)
		status = ca_get_callback(DBR_STS_STRING,pchandata->chid, 
			ca_get_dbr_sts_string_callback,pchandata);
		else
		status = ca_get_callback(DBR_CTRL_DOUBLE,pchandata->chid,
			ca_get_dbr_ctrl_double_callback,pchandata);

if (pchandata->type == DBR_GR_ENUM || pchandata->type ==  DBR_ENUM)  
	ca_get_conn_data(DBR_GR_ENUM,pchandata);

	        if (status != ECA_NORMAL) {
			command_error = CA_FAIL;
			ca_check_array_return_code(status);
				}
			}
                snode = snode->next;
		}
        	status = ca_pend_event(CA.PEND_EVENT_TIME);

fprintf(stderr,"\nDEVICE                       TYPE VALUE STATUS SEVR  UNITS       UOPR     LOPR \n");

        max = 10. / CA.PEND_EVENT_TIME;

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {

                        /* wait for ca_get_callback to finish */

                        while (pchandata->error == CA_WAIT) {
                                count++;
                                if (count > max) break;
                                ca_pend_event(CA.PEND_EVENT_TIME);
                                }

fprintf(stderr,"%-30s %2d %10f %2d %2d %-9s %10f %10f\n",
ca_name(pchandata->chid),pchandata->type,pchandata->value,pchandata->status,
pchandata->severity,pchandata->units,pchandata->uopr,pchandata->lopr);
			*(value+i) = pchandata->value;
			}
                snode = snode->next;
		i++;
                }

	ca_check_command_error(command_error);
}

/**************************************************
 *  populate information for a device array
 *************************************************/
void ca_populate_info_array(noName,pvName)
int noName;
char **pvName;
{
int i,status,command_error=0;
int count=0,max;
chandata *list;
chandata *snode;


	command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list;
        while (snode)  {
                pchandata = snode;
		if (pchandata->type != TYPENOTCONN) {
                ca_puser(pchandata->chid) = pchandata;
                pchandata->error = CA_WAIT;

		if (pchandata->type == DBR_STRING)
		status = ca_get_callback(DBR_STS_STRING,pchandata->chid, 
			ca_get_dbr_sts_string_callback,pchandata);
		else
		status = ca_get_callback(DBR_CTRL_DOUBLE,pchandata->chid,
			ca_get_dbr_ctrl_double_callback,pchandata);

if (pchandata->type == DBR_GR_ENUM || pchandata->type ==  DBR_ENUM)  
	ca_get_conn_data(DBR_GR_ENUM,pchandata);

	        if (status != ECA_NORMAL) {
			command_error = CA_FAIL;
			ca_check_array_return_code(status);
				}
			}
                snode = snode->next;
		}
        	status = ca_pend_event(CA.PEND_EVENT_TIME);

        max = 10. / CA.PEND_EVENT_TIME;

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {

                        /* wait for ca_get_callback to finish */

                        while (pchandata->error == CA_WAIT) {
                                count++;
                                if (count > max) break;
                                ca_pend_event(CA.PEND_EVENT_TIME);
                                }
			}
                snode = snode->next;
		i++;
                }

	ca_check_command_error(command_error);
}

/**************************************************
 *  get error code  for a device array
 *************************************************/
void ca_get_error_array(noName,pvName,value)
int noName;
char **pvName;
int  *value;
{
int i,status,command_error=0;
chandata *list;
chandata *snode,*phead;

        phead = (chandata *)&list;
        snode = phead;
        for (i=0;i<noName;i++) {
                pchandata = (chandata *)ca_check_hash_table(pvName[i]);
                snode->next = pchandata;
                pchandata->next = NULL;
                snode = pchandata;
                }
        status = ca_pend_io(CA.PEND_IOLIST_TIME);
        ca_check_array_return_code(status);

        snode = list; i=0;
        while (snode)  {
                pchandata = (chandata *)snode;
		pchandata->type = ca_field_type(pchandata->chid);
                if (pchandata->type == TYPENOTCONN) {
                         fprintf(stderr,"%-30s  ***  device not found\n",pvName[i]);
			command_error = CA_FAIL;
			*(value+i) = pchandata->type;
			}
                else {
			if (pchandata->error != 0) command_error = CA_FAIL;
			*(value+i) = pchandata->error;  /* timeout if is -2 */
			}
                snode = snode->next; i++;
                }

	ca_check_command_error(command_error);
}

/****************************************************
 *  get status value back from current data structure 
 *   -1 indicates unconnected
 ****************************************************/
void ca_old_status_array(noName,pvName,value)
int noName;
char **pvName;
int  *value;
{
int i,status,command_error=0;
chandata *list;
chandata *pnow,*phead;

        phead = (chandata *)&list;
        pnow = phead;
        for (i=0;i<noName;i++) {
                pchandata = (chandata *)ca_check_hash_table(pvName[i]);
                pnow->next = pchandata;
                pchandata->next = NULL;
                pnow = pchandata;
                }
        status = ca_pend_io(CA.PEND_IOLIST_TIME);
        ca_check_array_return_code(status);

        pnow = list; i=0;
        while (pnow)  {
                pchandata = pnow;
		pchandata->type = ca_field_type(pchandata->chid);
                if (pchandata->type == TYPENOTCONN) {
                         fprintf(stderr,"%-30s  ***  device not found\n",pvName[i]);
			command_error = CA_FAIL;
			*(value+i) = CA_FAIL;
			}
                else {
			*(value+i) = pchandata->status;
			}
                pnow = pnow->next; i++;
                }

	ca_check_command_error(command_error);
}

/**************************************************
 *  get status value back for a device array
 *************************************************/
void ca_get_status_array(noName,pvName,value)
int noName;
char **pvName;
int  *value;
{
int i,status,command_error=0;
int  count=0,max;
chandata *list;
chandata *snode;
struct dbr_sts_double dbr;

        command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {
                     ca_puser(pchandata->chid) = pchandata;
		     pchandata->error = CA_WAIT;
		     if (pchandata->type == DBR_STRING)  
		     status = ca_get_callback(DBR_STS_STRING,pchandata->chid, 
				ca_get_dbr_sts_string_callback,pchandata);
		else 	
	    	     status = ca_get_callback(DBR_STS_DOUBLE,pchandata->chid, 
				ca_get_dbr_sts_double_callback,pchandata);
		     if (status != ECA_NORMAL) {
				command_error = CA_FAIL;
                                ca_check_array_return_code(status);
				}
			}
                snode = snode->next;
                }

        max = 10. / CA.PEND_EVENT_TIME;

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {

                        while (pchandata->error == CA_WAIT) {
                                count++;
                                if (count > max) break;
                                ca_pend_event(CA.PEND_EVENT_TIME);
                                }
			*(value+i) = pchandata->status;
			}
                snode = snode->next; i++;
                }

	ca_check_command_error(command_error);
}

/**************************************************
 *  get double value back for a device array
 *************************************************/
void ca_get_double_array2(noName,pvName,value)
int noName;
char **pvName;
double *value;
{
int i,status,command_error=0;
void ca_get_dbr_sts_double_callback();
chandata *list;
chandata *snode;

        command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {
			if (pchandata->type == DBR_STRING)  
			status = ca_get_conn_data(DBR_STS_STRING,pchandata);
			else 	
			status = ca_get_conn_data(DBR_STS_DOUBLE,pchandata);
			if (status != ECA_NORMAL) command_error = CA_FAIL;
			*(value+i) = pchandata->value;
			}
                snode = snode->next; i++;
                }

	ca_check_command_error(command_error);
}

/**************************************************
 *  get double value,status,severity  back for a device array using callback
 *************************************************/
void ca_get_all_double_array(noName,pvName,value)
int noName;
char **pvName;
double *value;
{
int i,status,command_error=0;
int count=0,max;
chandata *list; 
chandata *snode;

        command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list;
        while (snode)  {
                pchandata = snode;
		if (pchandata->type != TYPENOTCONN) {
			ca_puser(pchandata->chid) = pchandata;
			pchandata->error = CA_WAIT;
			status = ca_get_callback(DBR_STS_DOUBLE,pchandata->chid,
				ca_get_dbr_sts_double_callback,pchandata);

			if (status != ECA_NORMAL) {
				 command_error = CA_FAIL;
				 ca_check_array_return_code(status);
				}
			}
                snode = snode->next;
                }
        status = ca_pend_event(CA.PEND_EVENT_TIME);

	max = 10. / CA.PEND_EVENT_TIME;

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {

			/* wait for ca_get_callback to finish */

			while (pchandata->error == CA_WAIT) {
				count++;
				if (count > max) break;
				ca_pend_event(CA.PEND_EVENT_TIME);
				}

			*(value+i*3) = pchandata->value;
			*(value+i*3+1) = pchandata->status;
			*(value+i*3+2) = pchandata->severity;
			}
                snode = snode->next; i++;
		}

	ca_check_command_error(command_error);
}

/**************************************************
 *  get double value back for a device array using callback
 *************************************************/
void ca_get_double_array(noName,pvName,value)
int noName;
char **pvName;
double *value;
{
int i,status,command_error=0;
int count=0,max;
chandata *list;
chandata *snode;

        command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list;
	
        while (snode)  {
                pchandata = snode;
/*fprintf(stderr,"%s, %x, next=%x\n",ca_name(pchandata->chid),pchandata,pchandata->next);
 */
		if (pchandata->type != TYPENOTCONN) {
			ca_puser(pchandata->chid) = pchandata;
			pchandata->error = CA_WAIT;
			status = ca_get_callback(DBR_STS_DOUBLE,pchandata->chid,
				ca_get_dbr_sts_double_callback,pchandata);

			if (status != ECA_NORMAL) {
				 command_error = CA_FAIL;
				 ca_check_array_return_code(status);
				}
			}
                snode = snode->next;
                }
        status = ca_pend_event(CA.PEND_EVENT_TIME);

	max = 10. / CA.PEND_EVENT_TIME;

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {

			/* wait for ca_get_callback to finish */

			while (pchandata->error == CA_WAIT) {
				count++;
				if (count > max) break;
				ca_pend_event(CA.PEND_EVENT_TIME);
				}

			*(value+i) = pchandata->value;
			}
                snode = snode->next; i++;
		}

	ca_check_command_error(command_error);
}


/***************************************************************
 * put string array for a device array
 *************************************************************/
void ca_put_string_array(noName,pvName,value)
int noName;
char **pvName;
char **value;
{
int i,status,command_error=0;
chandata *list;
chandata *snode;

        command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {
                if (strlen(value[i]) > 0) {
                        if (CA.devprflag > 0) 
			    fprintf(stderr,"%-30s %s\n",pvName[i],value[i]);
			status = ca_put(DBR_STRING,pchandata->chid,value[i]);
                        	}
                        }
                snode = snode->next; i++;
                }

	status = ca_pend_io(CA.PEND_IOLIST_TIME);

	if (status != ECA_NORMAL) { 
		command_error = CA_FAIL;
                ca_check_array_return_code(status);
		}

	ca_check_command_error(command_error);
}

/***************************************************************
 * put double value array for a device array
 *************************************************************/
void ca_put_double_array(noName,pvName,value)
int noName;
char **pvName;
double *value;
{
int i,status,command_error=0;
chandata *list;
chandata *snode;

        command_error = ca_pvlist_search(noName,pvName,&list);

        snode = list; i=0;
        while (snode)  {
                pchandata = snode;
                if (pchandata->type != TYPENOTCONN) {
                        if (CA.devprflag > 1)
				printf("%-30s %f\n",pvName[i],value[i]);
                        ca_put(DBR_DOUBLE,pchandata->chid,value+i); 
                        }
                snode = snode->next; i++;
                }

        status = ca_pend_io(CA.PEND_IOLIST_TIME);

        if (status != ECA_NORMAL) {
                command_error = CA_FAIL;
                ca_check_array_return_code(status);
                }

	ca_check_command_error(command_error);
}


/******************************************
 * get waveform data
 * return a float array, the first element is the no of elements
 *              in the array, followed with the waveform array values
 *****************************************/
double *ca_get_wave_form(name)
char *name;
{
double *ret;     /* return real array */
char string[40];
chid ix1,ix2;
int i,count=0,type,status,command_error=0;
unsigned size1,size2,size3;

ca_find_dev(name,pchandata);
 
if (pchandata->type != TYPENOTCONN) {
        type = ca_field_type(pchandata->chid);
        count = ca_element_count(pchandata->chid);

        ret = (double *)calloc(sizeof(double),count+2);

if(pchandata->state == cs_closed) {
	fprintf(stderr,"%s --- Invalid device name\n",ca_name(pchandata->chid));
	status = ECA_BADCHID;
	ca_check_return_code(status);
	}
else {
        size1 = dbr_size_n(type,count);
        size2 = dbr_size[type];
        size3 = dbr_value_size[type];
if (CA.devprflag > 0)
fprintf(stderr,"type=%d, count=%d, size1=%d, size2=%d, size3=%d\n",
        type,count,size1,size2,size3);

        ca_array_get(type,count,pchandata->chid,&wf_buf);
        status = ca_pend_io(CA.PEND_IO_TIME);
        if (status != ECA_NORMAL)  ca_check_return_code(status);
		

switch (type) {

case DBR_CHAR:                          /* UCHAR */
        for (i=0;i<count;i++) *(ret+i+1) = wf_buf.cv[i];
        break;

case DBR_STRING:                        /* STRING*/
        for (i=0;i<count;i++) *(ret+i+1) = atof(wf_buf.pv[i]);
        break;


case DBR_FLOAT:                         /* FLOAT */
        for (i=0;i<count;i++) *(ret+i+1) = wf_buf.fv[i];
        break;

case DBR_DOUBLE:                        /* DOUBLE */
        for (i=0;i<count;i++) *(ret+i+1) = wf_buf.dv[i];
        break;

case DBR_SHORT:                        /* SHORT*/
        for (i=0;i<count;i++) *(ret+i+1) = wf_buf.sv[i];
        break;

default: /* CHAR SHORT USHORT  LONG ULONG */
        for (i=0;i<count;i++) *(ret+i+1) = wf_buf.iv[i];
        }

if (CA.devprflag > 0) {
        fprintf(stderr,"ca_get_wave_form: ");
        for (i=1;i<=count;i++) fprintf(stderr,"%f ",*(ret+i));

        fprintf(stderr,"\n");
                }
    }
	}
        else { ret = (double *)calloc(sizeof(double),count+2); }

        *ret = count;
        return (ret);
}

/******************************************
 * put waveform data
 * return  n data (n<1000)
 *         CA_FAIL failed
 *****************************************/
int ca_put_wave_form(name,n,val)
char *name;
int n;
double *val;

{
char string[40];
chid ix1,ix2;
int i,count=0,type,status;
unsigned size1,size2,size3;

ca_find_dev(name,pchandata);
if (pchandata->type != TYPENOTCONN && pchandata->state != cs_closed) {
        type = ca_field_type(pchandata->chid);
        count = ca_element_count(pchandata->chid);

        size1 = dbr_size_n(type,count);
        size2 = dbr_size[type];
        size3 = dbr_value_size[type];
if (CA.devprflag > 0)
fprintf(stderr,"type=%d, count=%d, size1=%d, size2=%d, size3=%d\n",
        type,count,size1,size2,size3);

if (count > n) count=n;

if (CA.devprflag > 0) {
        fprintf(stderr,"ca_put_wave_form: ");
        for (i=0;i<count;i++) fprintf(stderr,"%f ",*(val+i));
        fprintf(stderr,"\n");
                }

switch (type) {

case DBR_CHAR:                          /* UCHAR */
        for (i=0;i<count;i++) wf_buf.cv[i] = *(val+i);
        ca_array_put(type,count,pchandata->chid,wf_buf.cv);
        break;

case DBR_STRING:                          /* STRING */
        for (i=0;i<count;i++) sprintf(wf_buf.pv[i],"%f", *(val+i));
        ca_array_put(type,count,pchandata->chid,wf_buf.pv);
        break;

case DBR_FLOAT:                         /* FLOAT */
        for (i=0;i<count;i++) wf_buf.fv[i] = *(val+i);
        ca_array_put(type,count,pchandata->chid,wf_buf.fv);
        break;

case DBR_DOUBLE:                        /* DOUBLE */
        for (i=0;i<count;i++) wf_buf.dv[i] = *(val+i);
        ca_array_put(type,count,pchandata->chid,wf_buf.dv);
        break;

case DBR_SHORT:                        /*  SHORT */
        for (i=0;i<count;i++) wf_buf.sv[i] = *(val+i);
        ca_array_put(type,count,pchandata->chid,wf_buf.sv);
        break;

default: /* CHAR UCHAR USHORT  LONG ULONG */
        for (i=0;i<count;i++) wf_buf.iv[i] = *(val+i);
        ca_array_put(type,count,pchandata->chid,wf_buf.iv);
        }

        status = ca_pend_io(CA.PEND_IO_TIME);
        if (status != ECA_NORMAL) ca_check_return_code(status);

        return (count);
	}
	
        else { return(CA_FAIL); }
}


