/* 
 * tclet.c --
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "tk.h"
#include "et_private.h"

#define CONNECTIONS_TOTAL  20

typedef struct connect_id_t {
  int	     index;
  et_sys_id  id;
  char	     name[ET_FILENAME_LENGTH];
} connect_id;

static connect_id cid[CONNECTIONS_TOTAL];
static int count=0;
int Tcl_AppInit();


/***********************************************************************
 *
 * Create a new ET system, specfying:
 *	1) the total number of events in the system
 *	2) the size of each normal event in bytes, and
 *	3) the name of the file to be used for memory mapping
 *
 ***********************************************************************/
 int et_create_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{  
  int i, numEvents=2000, eventSize=3000, index=-1;
  et_sysconfig  config;
  char *et_filename = NULL;
  char  et_name[ET_FILENAME_LENGTH];
  
  
  if (argc > 4) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " [number of events] [event size (bytes)] [et filename]", NULL);
    return TCL_ERROR;
  }
  
  if (count >= CONNECTIONS_TOTAL) {
    Tcl_AppendResult(interp, argv[0], ": no additional ET system connections allowed", NULL);
    return TCL_ERROR;
  }
  
  /* check for "number of events" argment */
  if (argc > 1) {
    if (Tcl_GetInt(interp, argv[1], &numEvents) == TCL_ERROR) {
      Tcl_AppendResult(interp, argv[0], ": \"number of events\" argument must be integer", NULL);
      return TCL_ERROR;
    }
    if (numEvents < 1) {
      Tcl_AppendResult(interp, argv[0], ": need at least 1 event in ET system", NULL);
      return TCL_ERROR;
    }
  }
  
  /* check for "event size" argment */
  if (argc > 2) {
    if (Tcl_GetInt(interp, argv[2], &eventSize) == TCL_ERROR) {
      Tcl_AppendResult(interp, argv[0], ": \"event size\" argument must be integer", NULL);
      return TCL_ERROR;
    }
    if (eventSize < 1) {
      Tcl_AppendResult(interp, argv[0], ": need at least 1 byte ET event size", NULL);
      return TCL_ERROR;
    }
  }
  
  /* check for "filename" argment */
  if (argc > 3) {
    if (strlen(argv[3]) > ET_FILENAME_LENGTH - 1) {
      Tcl_AppendResult(interp, argv[0], ": ET filename is too long", NULL);
      return TCL_ERROR;
    }
    strcpy(et_name, argv[3]);
  }
  /* if no "filename" argment ... */
  else {
    /* see if env variable SESSION is defined */
    if ( (et_filename = getenv("SESSION")) == NULL ) {
      Tcl_AppendResult(interp, argv[0], ": no filename given and SESSION env variable undefined", NULL);
      return TCL_ERROR;
    }
    /* check length of name */
    if ( (strlen(et_filename) + 12) >=  ET_FILENAME_LENGTH) {
      Tcl_AppendResult(interp, argv[0], ":  ET filename is too long", NULL);
      return TCL_ERROR;
    }
    sprintf(et_name, "%s%s", "/tmp/et_sys_", et_filename);
  }
  
  /* look for an unused id */
  for (i=0; i < CONNECTIONS_TOTAL; i++) {    
    if (cid[i].index < 0) {
      index = i;
      break;
    }
  }
  if (index < 0) {
    Tcl_AppendResult(interp, argv[0], ": no additional ET system connections allowed", NULL);
    return TCL_ERROR;
  }
  
  
  /* configure & start ET system  */
  if (et_system_config_init(&config) == ET_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": no more memory", NULL);
    return TCL_ERROR;
  }

  if (et_system_config_setfile(config, et_name) == ET_ERROR) {
    et_system_config_destroy(config);
    Tcl_AppendResult(interp, argv[0], ": bad filename argument", NULL);
    return TCL_ERROR;
  }
  
  if (et_system_config_setevents(config, numEvents) == ET_ERROR) {
    et_system_config_destroy(config);
    Tcl_AppendResult(interp, argv[0], ": \"number of events\" argument has bad value", NULL);
    return TCL_ERROR;
  }
  
  if (et_system_config_setsize(config, eventSize) == ET_ERROR) {
    et_system_config_destroy(config);
    Tcl_AppendResult(interp, argv[0], ": \"event size\" argument has bad value", NULL);
    return TCL_ERROR;
  }
  
  if (et_system_start(&cid[index].id, config) != ET_OK) {
    et_system_config_destroy(config);
    Tcl_AppendResult(interp, argv[0], ":  error in starting ET system", NULL);
    return TCL_ERROR;
  }
  et_system_config_destroy(config);
  
  /* do the bookkeeping */
  cid[index].index = index;
  strcpy(cid[index].name, argv[1]);
  count++;
  
  /* return a unique id */
  sprintf(interp->result, "%d", index);
  return TCL_OK;
}


/***********************************************************************
 *
 * Destroy a created ET system
 *
 * WARNING: In Linux, calling this will likely mess up your program
 * with a segmentation violation or something like that. It's
 * a problem inherint in the Linux pthreads library and I can
 * do nothing about it.
 *
 ***********************************************************************/
int et_destroy_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id;
  
  if (argc != 2) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et_id> ", NULL);
    return TCL_ERROR;
  }
  
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": argument value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
    
  if (et_system_close(cid[et_id].id) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": only the process that created the ET system can destroy it", NULL);
    return TCL_ERROR;
  }
  
  cid[et_id].index = -1;
  count--;
  return TCL_OK;
}


/***********************************************************************
 *
 * Connect to an existing ET system
 *
 ***********************************************************************/
int et_open_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{  
  int i, index=-1;
  struct timespec timeout;
  et_openconfig   openconfig;
  
  /* time to wait for connect to ET systems */
  timeout.tv_sec  = 30;
  timeout.tv_nsec = 0;
  
  if (argc != 2) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et filename> ", NULL);
    return TCL_ERROR;
  }
  
  if (count >= CONNECTIONS_TOTAL) {
    Tcl_AppendResult(interp, argv[0], ": no additional ET opens allowed", NULL);
    return TCL_ERROR;
  }
  
  if (strlen(argv[1]) > ET_FILENAME_LENGTH - 1) {
    Tcl_AppendResult(interp, argv[0], ": ET filename is too long", NULL);
    return TCL_ERROR;
  }
  
  /* if already connected return that id   */
  /* else, find first unused spot in array */
  for (i=0; i < CONNECTIONS_TOTAL; i++) {    
    if (cid[i].index > -1) {
      if (strcmp(argv[1], cid[i].name) == 0) {
        /* printf("et_connect: that connection already exists\n"); */
        sprintf(interp->result, "%d", i);
        return TCL_OK;
      }
    }
  }
  
  for (i=0; i < CONNECTIONS_TOTAL; i++) {    
    if (cid[i].index < 0) {
      index = i;
      break;
    }
  }
  
  if (index < 0) {
    Tcl_AppendResult(interp, argv[0], ": no additional ET opens allowed", NULL);
    return TCL_ERROR;
  }
  
  et_open_config_init(&openconfig);
  /* wait for "timeout" trying to connect */
  et_open_config_setwait(openconfig, ET_OPEN_WAIT);
  et_open_config_settimeout(openconfig, timeout);
  /* try to connect to local or remote host */
  et_open_config_sethost(openconfig, ET_HOST_ANYWHERE);
  if (et_open(&cid[index].id, argv[1], openconfig) != ET_OK) {
    et_open_config_destroy(openconfig);
    Tcl_AppendResult(interp, argv[0], ": cannot open ET system ", argv[1], NULL);
    return TCL_ERROR;
  }
  et_open_config_destroy(openconfig);
  
  cid[index].index = index;
  strcpy(cid[index].name, argv[1]);
  
  sprintf(interp->result, "%d", index);
  count++;
  return TCL_OK;
}


/***********************************************************************
 *
 * Disconnect from an existing ET system
 *
 ***********************************************************************/
int et_close_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id;
  
  if (argc != 2) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et_id> ", NULL);
    return TCL_ERROR;
  }
  
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": argument must be integer id", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": argument value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
    
  if (et_forcedclose(cid[et_id].id) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": cannot close ET system ", cid[et_id].name, NULL);
    return TCL_ERROR;
  }
  
  cid[et_id].index = -1;
  count--;
  return TCL_OK;
}


/***********************************************************************
 *
 * Attach to an existing station
 *
 ***********************************************************************/
int et_attach_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{  
  int et_id;  
  et_stat_id stat_id;
  et_att_id  att_id;
  
  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <station id> ", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer id", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
  
  /* station id */
  if (Tcl_GetInt(interp, argv[2], &stat_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": second argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (stat_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": station id value out-of-range", NULL);
    return TCL_ERROR;
  }
    
  if (et_station_attach(cid[et_id].id, stat_id, &att_id) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": cannot attach to station ",argv[2], NULL);
    return TCL_ERROR;
  }
    
  sprintf(interp->result, "%d", att_id);
  return TCL_OK;
}


/***********************************************************************
 *
 * Attach to an existing station (identified by name)
 *
 ***********************************************************************/
int et_attach2_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{  
  int et_id;  
  et_stat_id stat_id;
  et_att_id  att_id;
  
  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <station name> ", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer id", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
  
  /* given station name, get its id */
  if (et_station_name_to_id(cid[et_id].id, &stat_id, argv[2]) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": station ",argv[2], " does not exist", NULL);
    return TCL_ERROR;
  }
  
  if (et_station_attach(cid[et_id].id, stat_id, &att_id) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": cannot attach to station ",argv[2], NULL);
    return TCL_ERROR;
  }
    
  sprintf(interp->result, "%d", att_id);
  return TCL_OK;
}


/***********************************************************************
 *
 * Detach from an existing station
 *
 ***********************************************************************/
int et_detach_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id;  
  et_att_id  att_id;
  
  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <attach id> ", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": et id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
    
  /* attach id */
  if (Tcl_GetInt(interp, argv[2], &att_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": second argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (att_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": attach id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  if (et_station_detach(cid[et_id].id,  att_id) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": cannot detach from station", NULL);
    return TCL_ERROR;
  }
    
  return TCL_OK;
}


/***********************************************************************
 *
 * Create a new station
 *
 ***********************************************************************/
int et_stationcreate_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{  
  int et_id, blocking, cue, prescale;
  char *tmp;  
  et_statconfig sconfig;
  et_stat_id stat_id;
  
  if (argc < 3 || argc > 6) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <name> [blocking] [cue] [prescale]", NULL);
    return TCL_ERROR;
  }
  
  /* check ET system id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer id", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
  
  /* check station name */
  if (strlen(argv[2]) > ET_STATNAME_LENGTH - 1) {
    Tcl_AppendResult(interp, argv[0], ": station name is too long", NULL);
    return TCL_ERROR;
  }
  
  /* initialize station configuration */
  et_station_config_init(&sconfig);
  
  /* check 3rd arg to see if blocking or nonblocking */
  if (argc > 3) {
    if (Tcl_GetInt(interp, argv[3], &blocking) == TCL_ERROR) {
      /* it's not an int, so see if it's a string of "blocking" or "nonblocking" */
      if ((strcmp(argv[3], "blocking") == 0) ||
          (strcmp(argv[3], "BLOCKING") == 0))  {
        et_station_config_setblock(sconfig, ET_STATION_BLOCKING);
      }
      else if ((strcmp(argv[3], "nonblocking") == 0) ||
               (strcmp(argv[3], "NONBLOCKING") == 0))  {
        et_station_config_setblock(sconfig, ET_STATION_NONBLOCKING);
      }
      else {
        et_station_config_destroy(sconfig);
        Tcl_AppendResult(interp, argv[0], ": blocking argument must be 1 or blocking, OR 0 or nonblocking", NULL);
        return TCL_ERROR;
      }
    }
    else {
      if (blocking == 1) {
        et_station_config_setblock(sconfig, ET_STATION_BLOCKING);
      }
      else if (blocking == 0) {
        et_station_config_setblock(sconfig, ET_STATION_NONBLOCKING);
      }
      else {
        et_station_config_destroy(sconfig);
        Tcl_AppendResult(interp, argv[0], ": blocking argument must be 1 or blocking, OR 0 or nonblocking", NULL);
        return TCL_ERROR;
      }
    }
  }
  
  /* check 4th arg for cue size */
  if (argc > 4) {
    if (Tcl_GetInt(interp, argv[4], &cue) == TCL_ERROR) {
      et_station_config_destroy(sconfig);
      Tcl_AppendResult(interp, argv[0], ": cue size argument must be integer > 0", NULL);
      return TCL_ERROR;
    }
    
    if (cue < 1) {
      et_station_config_destroy(sconfig);
      Tcl_AppendResult(interp, argv[0], ": cue size argument must be integer > 0", NULL);
      return TCL_ERROR;
    }
    
    et_station_config_setcue(sconfig, cue);
  }
  
  /* check 5th arg for prescale value */
  if (argc > 5) {
    if (Tcl_GetInt(interp, argv[5], &prescale) == TCL_ERROR) {
      et_station_config_destroy(sconfig);
      Tcl_AppendResult(interp, argv[0], ": prescale argument must be integer > 0", NULL);
      return TCL_ERROR;
    }
    
    if (prescale < 1) {
      et_station_config_destroy(sconfig);
      Tcl_AppendResult(interp, argv[0], ": prescale argument must be integer > 0", NULL);
      return TCL_ERROR;
    }
    
    et_station_config_setprescale(sconfig, prescale);
  }
  
  
  /* create station */
  if (et_station_create(cid[et_id].id, &stat_id, argv[2], sconfig) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": cannot create station ", argv[2], NULL);
    return TCL_ERROR;
  }
  
  /* free up memory */
  et_station_config_destroy(sconfig);
  
  sprintf(interp->result, "%d", stat_id);
  return TCL_OK;
}


/***********************************************************************
 *
 * Remove an existing station
 *
 ***********************************************************************/
int et_stationremove_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id;  
  et_stat_id  stat_id;
  
  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <station id> ", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": et id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
  
  /* station id */
  if (Tcl_GetInt(interp, argv[2], &stat_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": second argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (stat_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": station id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  if (et_station_remove(cid[et_id].id,  stat_id) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": cannot remove station", NULL);
    return TCL_ERROR;
  }
    
  return TCL_OK;
}


/***********************************************************************
 *
 * Is this attachment attached to this station?
 *
 ***********************************************************************/
int et_isattached_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id, result;  
  et_stat_id  stat_id;
  et_att_id   att_id;
  
  if (argc != 4) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <station id> <attach id>", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": et id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
  
  /* station id */
  if (Tcl_GetInt(interp, argv[2], &stat_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": station id argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (stat_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": station id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* attach id */
  if (Tcl_GetInt(interp, argv[3], &att_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": attach id argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (att_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": attach id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  result = et_station_isattached(cid[et_id].id, stat_id, att_id);
  if (result == 0 || result == 1) {
    sprintf(interp->result, "%d", result);
    return TCL_OK;
  }
  
  Tcl_AppendResult(interp, argv[0], ": bad station or attachment id", NULL);
  return TCL_ERROR;
}


/***********************************************************************
 *
 * Get an existing event
 *
 ***********************************************************************/
int et_getevent_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id, mode, msec;  
  et_att_id att_id;
  et_event *pe;
  struct timespec time;
  long nsec_total;
  
  if (argc < 3 || argc > 5) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <attach id> [mode] [time (millisec}]", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": et id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
    
  /* attach id */
  if (Tcl_GetInt(interp, argv[2], &att_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": second argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (att_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": attach id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check 3rd arg for mode, default = sleep */
  mode = ET_SLEEP;
  if (argc > 3) {
      if ((strcmp(argv[3], "sleep") == 0) ||
          (strcmp(argv[3], "SLEEP") == 0))  {
        mode = ET_SLEEP;
      }
      else if ((strcmp(argv[3], "timed") == 0) ||
               (strcmp(argv[3], "TIMED") == 0))  {
        mode = ET_TIMED;
      }
      else if ((strcmp(argv[3], "async") == 0) ||
               (strcmp(argv[3], "ASYNC") == 0))  {
        mode = ET_ASYNC;
      }
      else {
        Tcl_AppendResult(interp, argv[0], ": mode argument must be sleep, timed, or asymc", NULL);
        return TCL_ERROR;
      }
  }
  
  /* check 4th arg for the time of "timed" mode, default = 5 sec */
  msec = 5000;
  if (argc > 4) {
    if (Tcl_GetInt(interp, argv[4], &msec) == TCL_ERROR) {
      Tcl_AppendResult(interp, argv[0], ": time argument must be integer", NULL);
      return TCL_ERROR;
    }

    if (msec < 0) {
      Tcl_AppendResult(interp, argv[0], ": time to wait must be >= 0 millisec", NULL);
      return TCL_ERROR;
    }
  }
  
  nsec_total = 1000000*msec;
  if (nsec_total >= 1000000000L) {
    time.tv_sec  = nsec_total/1000000000L;
    time.tv_nsec = nsec_total - time.tv_sec*1000000000L;
  }
  else {
    time.tv_nsec = nsec_total;
    time.tv_sec  = 0L;
  }
  
  if (et_event_get(cid[et_id].id,  att_id, &pe, mode, &time) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": error getting event", NULL);
    return TCL_ERROR;
  }
    
  sprintf(interp->result, "%d", pe);
  return TCL_OK;
}


/***********************************************************************
 *
 * Get a new event
 *
 ***********************************************************************/
int et_newevent_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id, mode, msec, size;  
  et_att_id att_id;
  et_event *pe;
  struct timespec time;
  long nsec_total;
  
  if (argc < 4 || argc > 6) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <attach id> <size (bytes)> [mode] [time (millisec}]", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": et id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
    
  /* attach id */
  if (Tcl_GetInt(interp, argv[2], &att_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": second argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (att_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": attach id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* size */
  if (Tcl_GetInt(interp, argv[3], &size) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": third argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (size < 1) {
    Tcl_AppendResult(interp, argv[0], ": event size must be > 0 bytes", NULL);
    return TCL_ERROR;
  }
  
  /* check 3rd arg for mode, default = sleep */
  mode = ET_SLEEP;
  if (argc > 4) {
      if ((strcmp(argv[4], "sleep") == 0) ||
          (strcmp(argv[4], "SLEEP") == 0))  {
        mode = ET_SLEEP;
      }
      else if ((strcmp(argv[4], "timed") == 0) ||
               (strcmp(argv[4], "TIMED") == 0))  {
        mode = ET_TIMED;
      }
      else if ((strcmp(argv[4], "async") == 0) ||
               (strcmp(argv[4], "ASYNC") == 0))  {
        mode = ET_ASYNC;
      }
      else {
        Tcl_AppendResult(interp, argv[0], ": mode argument must be sleep, timed, or asymc", NULL);
        return TCL_ERROR;
      }
  }
  
  /* check 4th arg for the time of "timed" mode, default = 5 sec */
  msec = 5000;
  if (argc > 5) {
    if (Tcl_GetInt(interp, argv[5], &msec) == TCL_ERROR) {
      Tcl_AppendResult(interp, argv[0], ": time argument must be integer", NULL);
      return TCL_ERROR;
    }

    if (msec < 0) {
      Tcl_AppendResult(interp, argv[0], ": time to wait must be >= 0 millisec", NULL);
      return TCL_ERROR;
    }
  }
  
  nsec_total = 1000000*msec;
  if (nsec_total >= 1000000000L) {
    time.tv_sec  = nsec_total/1000000000L;
    time.tv_nsec = nsec_total - time.tv_sec*1000000000L;
  }
  else {
    time.tv_nsec = nsec_total;
    time.tv_sec  = 0L;
  }
  
  if (et_event_new(cid[et_id].id,  att_id, &pe, mode, &time, size) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": error getting new event", NULL);
    return TCL_ERROR;
  }
    
  sprintf(interp->result, "%d", pe);
  return TCL_OK;
}


/***********************************************************************
 *
 * Put an existing or new event back into ET system
 *
 ***********************************************************************/
int et_putevent_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int et_id, address;  
  et_att_id att_id;
  et_event *pe;
  
  if (argc != 4) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <et id> <attach id> <event pointer>", NULL);
    return TCL_ERROR;
  }
  
  /* et id */
  if (Tcl_GetInt(interp, argv[1], &et_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": first argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if ((et_id < 0) || (et_id > CONNECTIONS_TOTAL-1)) {
    Tcl_AppendResult(interp, argv[0], ": et id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* check to see if connected first */
  if (cid[et_id].index == -1) {
    Tcl_AppendResult(interp, argv[0], ": no ET system at this id", NULL);
    return TCL_ERROR;
  }
    
  /* attach id */
  if (Tcl_GetInt(interp, argv[2], &att_id) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": second argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (att_id < 0) {
    Tcl_AppendResult(interp, argv[0], ": attach id value out-of-range", NULL);
    return TCL_ERROR;
  }
  
  /* event pointer */
  if (Tcl_GetInt(interp, argv[3], &address) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": address argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (address < 1) {
    Tcl_AppendResult(interp, argv[0], ": address must be > 0", NULL);
    return TCL_ERROR;
  }
  
  pe = (et_event *) address;
  
  if (et_event_put(cid[et_id].id,  att_id, pe) != ET_OK) {
    Tcl_AppendResult(interp, argv[0], ": error getting event", NULL);
    return TCL_ERROR;
  }
    
  /* sprintf(interp->result, "%d", pe); */
  return TCL_OK;
}


/***********************************************************************
 *
 * Given an event pointer, return the data buffer pointer
 *
 ***********************************************************************/
int et_getdata_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int address;  
  et_event *pe;
  void *data;
  
  if (argc != 2) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <event pointer>", NULL);
    return TCL_ERROR;
  }
    
  /* event pointer */
  if (Tcl_GetInt(interp, argv[1], &address) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": address argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (address < 1) {
    Tcl_AppendResult(interp, argv[0], ": address must be > 0", NULL);
    return TCL_ERROR;
  }
  
  pe = (et_event *) address;
  
  et_event_getdata(pe, &data);
      
  sprintf(interp->result, "%d", data);
  return TCL_OK;
}


/***********************************************************************
 *
 * Given an event pointer, get length of data in bytes
 *
 ***********************************************************************/
int et_getdatasize_cmd(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[])
{
  int address, len;  
  et_event *pe;
  void *data;
  
  if (argc != 2) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " <event pointer>", NULL);
    return TCL_ERROR;
  }
    
  /* event pointer */
  if (Tcl_GetInt(interp, argv[1], &address) == TCL_ERROR) {
    Tcl_AppendResult(interp, argv[0], ": address argument must be integer", NULL);
    return TCL_ERROR;
  }
  
  if (address < 1) {
    Tcl_AppendResult(interp, argv[0], ": address must be > 0", NULL);
    return TCL_ERROR;
  }
  
  pe = (et_event *) address;
  
  et_event_getlength(pe, &len);
      
  sprintf(interp->result, "%d", len);
  return TCL_OK;
}


/***********************************************************************
 *
 * TCL initialization for ET interface
 *
 ***********************************************************************/
int Et_Init(Tcl_Interp *interp) {
    
    int i;
    
    /* et stuff */
    for (i=0; i < CONNECTIONS_TOTAL; i++) {
      cid[i].index = -1;
    }

    /*  define TCL commands */
    Tcl_CreateCommand(interp, "et_create", et_create_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_destroy", et_destroy_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_open", et_open_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_close", et_close_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_attach", et_attach_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_attach_by_name", et_attach2_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_detach", et_detach_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_station_create", et_stationcreate_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_station_remove", et_stationremove_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_is_attached", et_isattached_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_get_event", et_getevent_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_new_event", et_newevent_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_put_event", et_putevent_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_get_data", et_getdata_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
    Tcl_CreateCommand(interp, "et_get_data_size", et_getdatasize_cmd,
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
     return TCL_OK;
}
