/*
 * File        :  spawn.c
 * Author      :  RJStrzelinski
 * Date        :  03-22-92
 *
 * Description :
 *    A function for simplified "spawning" of processes.  This is modelled 
 *    after the VMS Sys$Spawn routine in that it allows the calling program
 *    to attach to the stdin and stdout streams of the spawned process at 
 *    the same time.  The function returns the process id of the spawned
 *    process (or -1 if the spawn fails at the fork operation).
 *
 * Changes     :  04-01-92    RJS@BNL  Removed call to signal to setup handler
 *                                     for SIGPIPE - signal handling should be
 *                                     be done externally.
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>

pid_t spawn( fName, toChild, fromChild )
   const char  *fName;
   FILE        **toChild, **fromChild;
{
   extern void CatchPipeSignal();
   int         to[2], from[2], fstat;
   pid_t       pid;
/*
 * Create pipes for two-way communication, then fork...
 */
   pipe( to ); pipe( from );
   if ((pid = fork()) == 0)
   {
/*    The child process uses dup2 to attach its stdin and stdout streams
 *    to the pipe endpoint file descriptors stored in the to and from
 *    arrays.  These files are then closed before the exec takes place.
 *    Two of these files will be opened automatically by the spawned 
 *    process as stdin and stdout (to[0] and from[1], respectively).
 */
      dup2( to[0], 0 ); dup2( from[1], 1 );
      close( to[0] ); close( to[1] );
      close( from[0] ); close( from[1] );
      execlp( fName, fName, (char *) NULL );
   }
/*
 * The parent process closes off the unused (by it) ends of the pipes
 * and reopens the endpoints with which it will communicate with the
 * spawned process's stdin and stdout.
 */
   close( to[0] ); close( from[1] );
   *toChild = fdopen( to[1], "w" ); *fromChild = fdopen( from[0], "r" );
/*
 * Return the process id of the spawned process.
 */
   return pid;
}




