#include <synch.h>
#include "semaphore.h"


/*
 * function must be called prior to semaphore use.
 *
 */
void
semaphore_init (Semaphore * s)
{
  pthread_mutexattr_t attr;
  pthread_condattr_t cattr;

  pthread_mutexattr_init(&attr);
  pthread_mutexattr_setpshared(&attr,PTHREAD_PROCESS_SHARED);
  pthread_condattr_init(&cattr);
  pthread_condattr_setpshared(&cattr,PTHREAD_PROCESS_SHARED);
  
  s->v = 1;
  if (pthread_mutex_init (&(s->mutex), &attr) == -1)
    do_error ("Error setting up semaphore mutex");
  
  if (pthread_cond_init (&(s->cond), &cattr) == -1)
    do_error ("Error setting up semaphore condition signal");
  s->i = 1;
}

/*
 * function should be called when there is no longer a need for
 * the semaphore.
 *
 */
void
semaphore_destroy (Semaphore * s)
{
    if (pthread_mutex_destroy (&(s->mutex)) == -1)
     do_error ("Error destroying semaphore mutex");

    s->i = 0;

    pthread_cond_broadcast(&(s->cond));

    if (pthread_cond_destroy (&(s->cond)) == -1)
      do_error ("Error destroying semaphore condition signal");
    
}

/*
 * function increments the semaphore and signals any threads that
 * are blocked waiting a change in the semaphore.
 *
 */
int
semaphore_give (Semaphore * s)
{
    int         value_after_op;

    tw_mutex_lock (&(s->mutex));
    (s->v)++;
    value_after_op = s->v;

    tw_mutex_unlock (&(s->mutex));
    tw_cond_signal (&(s->cond));

    return 0;
}

/*
 * function decrements the semaphore and blocks if the semaphore is
 * <= 0 until another thread signals a change.
 *
 */
int
semaphore_take (Semaphore * s)
{
    int         value_after_op;

    tw_mutex_lock (&(s->mutex));

    while (s->v <= 0)
    {
     pthread_cond_wait (&(s->cond), &(s->mutex));
     if (s->i == 0) {
       return -1;
     }
    }

    (s->v)--;
    value_after_op = s->v;

    tw_mutex_unlock (&(s->mutex));

    return 0;
}

/*
 * function does NOT block but simply decrements the semaphore.
 * should not be used instead of down -- only for programs where
 * multiple threads must up on a semaphore before another thread
 * can go down, i.e., allows programmer to set the semaphore to
 * a negative value prior to using it for synchronization.
 *
 */
int
semaphore_decrement (Semaphore * s)
{
    int         value_after_op;

    pthread_mutex_lock (&(s->mutex));
    s->v--;
    value_after_op = s->v;
    pthread_mutex_unlock (&(s->mutex));

    return 0;
}

int
semaphore_increment (Semaphore * s)
{
    int         value_after_op;
    pthread_mutex_lock (&(s->mutex));
    s->v++;
    value_after_op = s->v;
    pthread_mutex_unlock (&(s->mutex));
    return 0;
}

int
semaphore_set (Semaphore * s,int value)
{
    int         value_after_op;

    pthread_mutex_lock (&(s->mutex));
    s->v = value;
    value_after_op = s->v;
    pthread_mutex_unlock (&(s->mutex));

    return 0;
}

/*
 * function returns the value of the semaphore at the time the
 * critical section is accessed.  obviously the value is not guarenteed
 * after the function unlocks the critical section.  provided only
 * for casual debugging, a better approach is for the programmar to
 * protect one semaphore with another and then check its value.
 * an alternative is to simply record the value returned by semaphore_up
 * or semaphore_down.
 *
 */
int
semaphore_value (Semaphore * s)
{
    /* not for sync */
    int         value_after_op;

    pthread_mutex_lock (&(s->mutex));
    value_after_op = s->v;
    pthread_mutex_unlock (&(s->mutex));
    return (value_after_op);
}

/* -------------------------------------------------------------------- */
/* The following functions replace standard library functions in that   */
/* they exit on any error returned from the system calls.  Saves us     */
/* from having to check each and every call above.                      */
/* -------------------------------------------------------------------- */


int
tw_mutex_unlock (pthread_mutex_t * m)
{
    int         return_value;

    if ((return_value = pthread_mutex_unlock (m)) == -1)
     do_error ("pthread_mutex_unlock");

    return (return_value);
}

int
tw_mutex_lock (pthread_mutex_t * m)
{
    int         return_value;

    if ((return_value = pthread_mutex_lock (m)) == -1)
     do_error ("pthread_mutex_lock");

    return (return_value);
}

int
tw_cond_wait (pthread_cond_t * c, pthread_mutex_t * m)
{
    int         return_value;
    if ((return_value = pthread_cond_wait (c, m)) != 0)
      do_error ("pthread_cond_wait");
    return (return_value);
}

int
tw_cond_signal (pthread_cond_t * c)
{
    int         return_value;
    if ((return_value = pthread_cond_signal (c)) == -1)
     do_error ("pthread_cond_signal");
    return (return_value);
}

int 
tw_cond_dump (pthread_cond_t *c) 
{
  int i;
  long *p = (long *) c;

  for (i=0;i<sizeof(pthread_cond_t)/4;i++) {
    printf("%d: %08x\n",i,p[i]);
  }
  return 0;
}
/*
 * function just prints an error message and exits 
 *
 */
void
do_error (char *msg)
{
    perror (msg);
    exit (1);
}
