// 
//  dd_watch
//
//  creates and keeps dd system if master flag (-m) specified
//  prints dd stats to stdout -p specified
//  checks every check_time seconds (-c tsec)
//    
//  note...only 1 master dd_watch allowed per dd system per node
//
//  still to do:
//     signal handler
//
//  ejw, 11-mar-98


// for posix
#define _POSIX_SOURCE_ 1
#ifndef __EXTENSIONS__
# define __EXTENSIONS__
#endif

// for i/o
#include <fstream.h>
#include <iomanip.h>

// system stuff
#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <string.h>
#include <stdarg.h>

// for dd
extern "C" {
#include <dd_sys.h>
#include <dd_user.h>
#include <dd_config.h>
#include <dd_dcom.h>
#include <fifo.h>
}
struct dd_dcom_str *dd_dcom;

// misc variables
static int master             = 0;
static int print              = 1;
static int check_time         = 15;
static char *host             = getenv("HOST");
static int pid                = getpid();
static int done               = 0;
static char temp[128];

// dd session name 
static char *session          = getenv("DD_NAME");

// for log file
ofstream logfile;

// prototypes
void decode_command_line(int argc, char **argv);
void init_dd(void);
void print_dd_stats(void);
void dd_done(void);
void siginthandler(int sig);

// macros
#define	min(a, b)  ((a) > (b) ? (b) : (a))

//--------------------------------------------------------------------------


main(int argc,char **argv){

  int status,err;
  char *name;
  char pidc[10];
  time_t start=time(NULL);


  // synch with stdio
  ios::sync_with_stdio();


  // decode command line
  decode_command_line(argc,argv);


  // session name
  if(session==NULL)session="clasprod";


  // if master, open log file and print startup information
  if(master!=0) {
    strcpy(temp,"dd_watch.log");
    logfile.open(temp,ios::out|ios::app);
    logfile << endl << endl 
	    << "dd_watch" << " starting in session: " << session << ", on " << ctime(&start);  
  }		


  // initialize DD system
  init_dd();
  
  // sig handler
  signal(SIGINT, siginthandler);


  // main loop
  while ( ((status=ddu_attached())!=-1) && (done==0) ){
    if(print!=0)print_dd_stats();
    sleep(check_time);
  }


  // done
  dd_done();
  exit(EXIT_SUCCESS);
}
       

//--------------------------------------------------------------------------


void init_dd(){
  
  struct fifo_mode fmode;
  int ctl[FIFO_HDR_CTL_LEN]={-1,-1,-1,-1};
  char ftype[5];
  int status;
  
  
  // create dd system if master, otherwise just attach;  die on failure
  if(master!=0){
    if((status=dds_create())!=0){
      cerr << "\n?unable to create dd system, status = " << status << endl << endl;
      if(master!=0)logfile << "\n?unable to create dd system, status = " << status << endl << endl;
      exit(EXIT_FAILURE);
    }
  } else if((status=ddu_attach())!=0){
    cerr << "\n?unable to attach to dd system, status = " << status << endl << endl;
    exit(EXIT_FAILURE);
  }
  
  return;
}


//--------------------------------------------------------------------------


void print_dd_stats(void){


  int fstatus[MAX_FIFO_NB_MAX];
  int fcnt[MAX_FIFO_NB_MAX];
  int devin[MAX_FIFO_NB_MAX];
  int devout[MAX_FIFO_NB_MAX];
  static time_t last = (time_t) 0;
  int deltat = 2147483648;
  time_t now=time(NULL);
  char *t, *temp;
  int i,status,tlen;


  // don't know why this happens...
  if(dd_dcom==NULL){
    done=1;
    cerr << "?print_dd_stats...no dd_dcom!" << endl;
    if(master!=0)logfile << "?print_dd_stats...no dd_dcom!" << endl;
    return;
  }


  // don't know why this happens either...
  if((status=ddu_attached())==-1){
    done=1;
    cerr << "?print_dd_stats...bad status from ddu_attached: " << status << endl;
    if(master!=0)logfile << "?print_dd_stats...bad status from ddu_attached: " << status << endl;
    return;
  }


  // get elapsed time since last call
  deltat=(int)(now-last);
  last=now;

  // get fifo sem counts, rates, etc.
  fifo_rate(fstatus,fcnt,devin,devout);
    

  t=ctime(&now);
  t[strlen(t)-1]='\0';
  cout << endl << endl;
  cout << "-------- Status for DD system: " << session << "  pid: "
       << pid << "  host: " << host << "  on: " << t 
       << " --------" 
       << endl << endl;
  

  // dd system stuff
  cout << setw(17) << "DD status" << setw(11) << "nmstr" << setw(11) << "nproc"
       << setw(11) << "max_fifo" << setw(12) << "nfev_req" << setw(12) << "nbuf_made" 
       << setw(12) << "nbuf_vol" << endl;
  cout << setw(17) << dd_dcom->dd_status << setw(11) << dd_dcom->dd_nmstr
       << setw(11) << dd_dcom->dd_nproc_att << setw(11) << dd_dcom->fifo_max_in_use 
       << setw(12) << dd_dcom->nfev_req << setw(12) << dd_dcom->nbuf_vol_made
       << setw(12) << dd_dcom->db_nbuf_vol << endl;
  cout << endl << endl;
  
  
  // dd master stuff
  cout << setw(17) << "Master PID" << setw(11) << "sem" << setw(11) << "semval" << endl;
  for (i=0; i<dd_dcom->dd_nmstr; i++){
    cout << setw(17) << dd_dcom->dd_master[i].pid << setw(11) << dd_dcom->dd_master[i].dd_sem 
	 << setw(11) << dd_dcom->dd_master[i].dd_semval << endl;
  }
  cout << endl << endl;


  // dd config stuff
  cout << setw(17) << "DD max_fifo_nb" << setw(11) << "fifo_depth" << setw(11) << "maxnbev"
       << setw(11) << "maxevsize" << setw(11) << "maxbufvol" << endl;
  cout << setw(17) << dd_dcom->dd_config.max_fifo_nb 
       << setw(11) << dd_dcom->dd_config.fifo_depth
       << setw(11) << dd_dcom->dd_config.db_maxnbev
       << setw(11) << dd_dcom->dd_config.db_maxevsize
       << setw(11) << dd_dcom->dd_config.db_max_buf_vol << endl;
  cout << endl << endl;
  

  // proc stuff
  cout << setw(28) << "DD proc name" << setw(11) << "PID" << setw(11) << "fifo"
       << setw(11)  << "wait" << setw(11) << "status" << setw(11) << "master"
       << setw(12)  << "evput" << setw(12) << "evget" << endl;
  for (i=0; i<dd_dcom->dd_nproc_att; i++){
    tlen=min(strlen(dd_dcom->dd_proc[i].pname),28);
    temp=(char *)malloc(tlen+2);
    strncpy(temp,dd_dcom->dd_proc[i].pname,tlen);
    temp[tlen]='\0';
    cout << setw(28) << temp << setw(11) << dd_dcom->dd_proc[i].pid
	 << setw(11)  << dd_dcom->dd_proc[i].fifo << setw(11) << dd_dcom->dd_proc[i].dd_wait
	 << setw(11)  << dd_dcom->dd_proc[i].status << setw(11) << dd_dcom->dd_proc[i].master_id
	 << setw(12)  << dd_dcom->dd_proc[i].nputfev << setw(12) << dd_dcom->dd_proc[i].ngetfev 
	 << endl;
    free(temp);
  }
  cout << endl << endl;



  // fifo system stuff
  cout << setw(17) << "DD sys name" << setw(11) << "PID_creat" << setw(11) << "status"
       << setw(11) << "max_fifo" << setw(11) << "depth"
       << setw(16) << "create time" << endl;
  cout << setw(17) << fifo_sys_start->sys_name << setw(11) << fifo_sys_start->pid_creat
       << setw(11) << fifo_sys_start->status             
       << setw(11) << fifo_sys_start->max_fifo_nb        
       << setw(11) << fifo_sys_start->fifo_depth         
       << "     " << ctime(&fifo_sys_start->time_creat);
  cout << endl << endl;

  
  // individual fifo info
  cout << setw(10) << "Fifo name" 
       << setw(4) << "num"  
       << setw(5) << "type" 
       << setw(5) << "stat" 
       << setw(5) << "mode" 
       << setw(5) << "proc" 
       << setw(5) << "wait"
       << setw(5) << "prot"
       << setw(5) << "freq" 
       << setw(7) << "prescl" 
       << setw(7) << "pid" 
       << setw(4) << "sem" << setw(4) << "cnt"
       << setw(10) << "optry" << setw(10) << "opin" << setw(10) << "opout" 
       << setw(7) << "devin" << setw(7) << "devout" 
       << setw(5) << "rin" << setw(5) << "rout" << endl;
  
  for(i=0; i<fifo_sys_start->max_fifo_nb; i++){
    struct a_fifo *pp;
    pp=fifo_shm_start +i;
    status=pp->fhdr.fstatus;
    if(fstatus[i]!=0){
      cout << setw(10) << pp->fhdr.fname << setw(4) << pp->fhdr.fnumber
	   << setw(5) << pp->fhdr.ftype << setw(5) <<  pp->fhdr.fstatus
	   << setw(5) << pp->fhdr.fmode 
	   << setw(5) << pp->fhdr.fnproc_att 
	   << setw(5) << pp->fhdr.fwait_mode
	   << setw(5) << pp->fhdr.fprotection
	   << setw(5) << pp->fhdr.freqcnt
	   << setw(7) << pp->fhdr.fprescale
	   << setw(7) << pp->fhdr.fpidcreat 
	   << setw(4) << fcnt[i] << setw(4) << pp->cnt
	   << setw(10) << pp->noptry << setw(10) << pp->nopin << setw(10) << pp->nopout 
	   << setw(7) << devin[i] << setw(7) << devout[i] 
	   << setw(5) <<  devin[i]/deltat << setw(5) << devout[i]/deltat << endl;
    }
  }
  cout << endl;
  
  return;
}


//-------------------------------------------------------------------


void dd_done(void){

  int status;
  time_t endtime=time(NULL);

  // shut down dd system if master, otherwise just deattach
  if(master!=0){
    logfile << "...master calling dds_close" << endl;
    dds_close(0);
  } else {
    ddu_close();
  }


  // print shutdown message and close log file
  if(master!=0){
    logfile << "dd_done...dd_watch stopping on " << ctime(&endtime) << endl << endl;
    logfile.close();
  }


  return;
}


//-------------------------------------------------------------------


void decode_command_line(int argc, char**argv){

  char *help = "\nusage:\n\n  dd_watch [-c check_time] [-m(aster)]\n\n";


  // loop over all arguments, except the 1st (which is program name)
  int i=1;
  while(i<argc) {
    if(strncasecmp(argv[i],"-h",2)==0){
      cout << help << endl;
      exit(EXIT_SUCCESS);
    }
    else if (strncasecmp(argv[i],"-m",2)==0){
      master=1;
      i=i+1;
    }
    else if (strncasecmp(argv[i],"-c",2)==0) {
      sscanf(argv[i+1],"%d",&check_time);
      i=i+2;
    }
    else if (strncasecmp(argv[i],"-",1)==0) {
      cout << "Unknown command line arg: " << argv[i] << argv[i+1] << endl << endl;
      i=i+2;
    }
  }

  return;
}

  
//----------------------------------------------------------------


void siginthandler (int sig) {
  cout << "\n    ...received SIGINT...shutting down..." << endl << endl;
  done=1;
  return;
}


//----------------------------------------------------------------

