/*
 * dd_garbage_d.c
 *
 * Author   :  C.Witzig
 * Date     :  June 15, 1992
 *
 * Modifications:
 * Dec 6, 1993: introduced environement variable DD_GARBAGE_D_LOGFILE
 *              to handle logfile properly. CW
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/signal.h>
#include <time.h>
#include <sys/time.h>

#include <dd_user.h>
#include <dd_sys.h>
#include <dd_signal_def.h>

#define TRUE 1
#define FALSE 0
#define TSEC_UPDATE  2
#define TUSEC_UPDATE 0

#define MAX_SEM_CNT 5
#define MAX_SUSPICIOUS_COUNT 25

static int timer_seen = FALSE;
static int stop_signal_seen = FALSE;
static dd_unused = FALSE;
static int suspicious_fifo[MAX_FIFO_NB_MAX];
FILE *logfile;

static char *status_filename;
static FILE *status_file;

static void stop_timer(), start_timer();


/*
 * Monitoring the DD system:
 * The garbage_demon also writes some monitoring information into
 * a DD_STATUS_FILE. Routines:
 * - open_status_file
 * - close_status_file
 * - update_status_file
 */

int open_status_file()
{
  size_t len;
  char *p2f, *p2dd;

  status_filename = NULL;

  if ( (p2f = getenv("DD_STATUS_FILE")) == NULL ) {
    return -1;
  }

  if ( (p2dd = getenv("DD_NAME")) == NULL )
    p2dd = DEFAULT_DD_NAME;

  len = strlen(p2f) + strlen(p2dd) + sizeof(pid_t) + 1;
  if ( (status_filename = malloc(len)) == NULL )
    return -1;

  sprintf(status_filename, "%s.%s", p2f, p2dd);

  if ( (status_file = fopen(status_filename, "w")) == NULL ) {
    /* free mem & set to NULL so no refrees of mem CTCTCT */
    free(status_filename);
    status_filename = NULL;
    return -1;
  }

  fclose(status_file);

  return 0;
}
  

int close_status_file()
{
  if ( status_filename == NULL )
    return;

  if ( unlink(status_filename) ) {
    sprintf("dd_garbage_d: pid %d - cannot unlink status_file %s\n",status_filename);
    return -1;
  }

  free(status_filename);

  return 0;
}

int update_status_file()
{
  int i, nproc, irate;
  static char *p2dd, *p2host;
  static int evt;
  float rate;

  if ( status_filename == NULL )
    return -1;

  if ( p2dd == NULL ) {
    if ( (p2dd = getenv("DD_NAME")) == NULL ) 
      p2dd = DEFAULT_DD_NAME;
  }

  if ( p2host == NULL ) {
    if ( (p2host = getenv("HOST")) == NULL ) 
      p2host = "unknown";
  }

  nproc = 0;
  for (i=0; i<DD_MAX_NPROC_ATT; i++) {
    if ( dd_dcom->dd_proc[i].pid > 0 )
      nproc++;
  }

  lock_input_fifo();
  rate = ( (float) (fifo_shm_start->nopin - evt)) / (TSEC_UPDATE + (1e-6)*TUSEC_UPDATE);
  evt = fifo_shm_start->nopin;
  unlock_input_fifo();

  if ( (status_file = fopen(status_filename, "w")) == NULL )
    return -1;

  fprintf(status_file, "%s %s %s %d %d", p2dd, p2host, dd_states[dd_dcom->dd_status], nproc, (int)rate);

  fclose(status_file);

  return 0;
}
  


/*
 * remove dead processes / fifos
 */
static void remove_dd_proc(int iproc)
{
  int i;

  dd_dcom_lock();

  if ( dd_dcom->dd_proc[iproc].pid > 0 ) {
    if ( dd_dcom->dd_proc[iproc].master_id >= 0 ) {
      for (i=0; i<DD_MAX_NMSTR; i++){
	if ( dd_dcom->dd_master[i].pid ==  dd_dcom->dd_proc[iproc].pid ) {
	  dd_dcom->dd_master[i].pid = -1;
	  dd_dcom->dd_nmstr--;
	}
      }
    }
    dd_dcom->dd_proc[iproc].pid  = -1;
    dd_dcom->dd_proc[iproc].fifo = -1;
    dd_dcom->dd_proc[iproc].status = DD_PROC_STATE_UNUSED;
    dd_dcom->dd_proc[iproc].nputfev = 0;
    dd_dcom->dd_proc[iproc].ngetfev = 0;
    strcpy(dd_dcom->dd_proc[iproc].pname,"");
    dd_dcom->dd_proc[iproc].master_id = -1;
    
    dd_dcom->dd_nproc_att--;

    dd_dcom_unlock();
  }
  return;
}


static void fix_fifo(int ififo)
{
/*
 * attempts to fix the fifo which has 
 * events in it but noone reads them 
 * for an extended period of time
 */
  int i, nproc, status, proc_list[DD_MAX_NPROC_ATT];

  fprintf(logfile,"fifo %d seems to be dead\n",ififo);

  nproc = 0;
  for (i=0; i<DD_MAX_NPROC_ATT; i++){
    if ( dd_dcom->dd_proc[i].pid > 0 ) {
      if ( dd_dcom->dd_proc[i].fifo == ififo ) {
	proc_list[nproc++] = i;
	fprintf(logfile,"fix_fifo: DD_SHUTDOWN_SIGNAL sent to process %d attached to fifo %d\n",dd_dcom->dd_proc[i].pid,ififo);
	if ( (status = kill(dd_dcom->dd_proc[i].pid, DD_SHUTDOWN_SIGNAL) ) ) {
	  if ( (status == ESRCH) )
	    remove_dd_proc(i);
	  else
	    fprintf(logfile,"fix_fifo: dd_garbage_d: error in removing process %d on fifo %d\n",
		   dd_dcom->dd_proc[i].pid,ififo);
	}
      }
    }
  }
  dd_nap(1.0);
  for (i=0; i<nproc; i++) {
    if ( dd_dcom->dd_proc[proc_list[i]].pid > 0 ) {
      if ( dd_dcom->dd_proc[proc_list[i]].fifo == ififo ){ 
/*	kill(dd_dcom->dd_proc[proc_list[i]].pid,SIGKILL);  
	fprintf(logfile,"fix_fifo: killing process %d attached to fifo %d\n",dd_dcom->dd_proc[i].pid,ififo); */
	remove_dd_proc(proc_list[i]);
      }
    }
  }
  
  fifo_set_status(ififo,FSTAT_IDLE);
  dbfi_empty_fifo(ififo);
  fifo_remove(ififo);

  return;
}

static int find_dd_zombies()
{
/*
 * checks the fifo rate and content of
 * unprotected fifos to detect any
 * hanging processes
 */
  struct a_fifo *pfifo;
  int i;
  int cnt[MAX_FIFO_NB_MAX], dnevout[MAX_FIFO_NB_MAX];
  static int nevoldout[MAX_FIFO_NB_MAX];

  for (i=0; i<max_fifo_nb; i++){
    pfifo = fifo_shm_start + i;
    cnt[i] = pfifo->cnt;
    dnevout[i]   = pfifo->nopout -  nevoldout[i];
    nevoldout[i] =  pfifo->nopout;
  }

  for (i=0; i<max_fifo_nb; i++) {
    pfifo = fifo_shm_start + i;
    if ( pfifo->fhdr.fprotection == FIFO_UNPROTECTED ) {
      if ( ( pfifo->cnt > MAX_SEM_CNT ) && (dnevout[i] == 0 ) )
	suspicious_fifo[i]++;
      else
	suspicious_fifo[i] = 0;
      if ( suspicious_fifo[i] > MAX_SUSPICIOUS_COUNT ) 
	fix_fifo(i);
    }
  }
  return 0;
}


static void check_dd_sem()
{
/*
 * makes sure that all registered processes have their
 * DD semaphore set. Loops first over the masters
 * then over all DD processes
 */
  int i, semcnt[MAX_DD_SEM];

  dd_dcom_lock();

  dd_dcom_get_semcnt(semcnt);

  for (i=0; i<DD_MAX_NMSTR; i++) {
    if ( dd_dcom->dd_master[i].pid > 0 ) {
      if ( (semcnt[dd_dcom->dd_master[i].dd_sem] & dd_dcom->dd_master[i].dd_semval) == 0 ) {
	fprintf(logfile,"check_dd_sem: removing master #%d pid %d from DD\n",i,dd_dcom->dd_master[i].pid);
	dd_dcom->dd_master[i].pid = -1;
	dd_dcom->dd_master[i].dd_sem = -1;
	dd_dcom->dd_master[i].dd_semval = -1;
    	dd_dcom->dd_nmstr--;
      }
    }
  }
	
  for (i=0; i<DD_MAX_NPROC_ATT; i++) {
    if ( dd_dcom->dd_proc[i].pid > 0 ) {
      if ( (semcnt[dd_dcom->dd_proc[i].dd_sem] & dd_dcom->dd_proc[i].dd_semval) == 0 ) {
	fprintf(logfile,"check_dd_sem: removing proc #%d pid %d from DD\n",i,dd_dcom->dd_proc[i].pid);

	if ( (dd_dcom->dd_proc[i].fifo > 0 ) && ( dd_dcom->dd_proc[i].fifo != INPUT_FIFO ) ) {
	  fifo_set_status(i,FSTAT_IDLE);
	  dbfi_empty_fifo(i);
	  fifo_remove(i);
	}

	dd_dcom->dd_proc[i].pid  = -1;
	dd_dcom->dd_proc[i].fifo = -1;
	dd_dcom->dd_proc[i].status = DD_PROC_STATE_UNUSED;
	dd_dcom->dd_proc[i].nputfev = 0;
	dd_dcom->dd_proc[i].ngetfev = 0;
	strcpy(dd_dcom->dd_proc[i].pname,"");
	dd_dcom->dd_proc[i].dd_sem = -1;
	dd_dcom->dd_proc[i].dd_semval = -1;
	dd_dcom->dd_nproc_att--;
      }
    }
  }

  dd_dcom_unlock();

  return;
}



/*
 * timer and other signals
 */

static void stop_timer()
{
  struct itimerval tto, tti = {0,0};

  if (setitimer(ITIMER_REAL, &tti, &tto))
    fprintf(logfile," error in setitimer %s \n",strerror(errno));
  
  return;
}


static void stop_signal(int sig)
{
  stop_timer();
  signal(DD_SHUTDOWN_SIGNAL, stop_signal);
  stop_signal_seen = TRUE;

  fprintf(logfile,"dd_garbage_d: stop signal seen\n");
  return;
}


static void check_eodd(int sig)
{
/*
 * check whether we have dead processes and whether
 * to close the DD system because nobody uses it anymore
 */

  stop_timer();

  timer_seen = TRUE;
  
  check_dd_sem();
  find_dd_zombies();
  if ( dd_dcom->dd_status == DD_STATUS_FKDUP )
    dd_error_inv();

  if ( dd_dcom->dd_nmstr == 1 ) {
    stop_timer();
    dd_unused = TRUE;
  }

  fflush(logfile);

  update_status_file();

  start_timer();

  return;
}

static void start_timer()
{
  static struct itimerval tto, tti = {TSEC_UPDATE,TUSEC_UPDATE, 
					TSEC_UPDATE,TUSEC_UPDATE};
  signal(SIGALRM,check_eodd);
  
  if (setitimer(ITIMER_REAL, &tti,&tto)){
    fprintf(logfile," error in setitimer %s \n",strerror(errno));
    exit(1);
  }
  return;
}


/*
 ************************************
 *              main                *
 ************************************
 */
main()
{
  time_t t;
  char fname[100];
  char logfilename[1024], *p2c, *p2host;
  int i, error, status;
  int ctl[FIFO_HDR_CTL_LEN];
  struct fifo_entry fev;
  struct fifo_mode fmode;
  int log_ok=1;
  pid_t pid;

  dd_nap(0.4);
  pid=getpid();

  p2c = getenv("DD_LOGFILES_DIRECTORY");
  if (p2c == NULL) {
    p2c = "/tmp";
    syslog(LOG_ERR,"dd_garbage_d %i: log directory not found - try /tmp\n", pid);
  }

  p2host = getenv("HOST");
  if (p2host == NULL) {
    syslog(LOG_ERR,"dd_garbage_d %i: HOST is not set. Quit garbage daemon!\n", pid);
    exit(1);
  }

  sprintf(logfilename, "%s/%s.%s.%d",p2c,"dd_garbage_log",p2host,getpid());

  if ((logfile = fopen(logfilename,"w")) == NULL) {
    syslog(LOG_ERR,"dd_garbage_d %i: can't open logfile, try /tmp directory\n", pid);
    sprintf(logfilename, "/tmp/dd_garbage_log.%s.%d",p2host,pid);
    
    if ((logfile = fopen(logfilename,"w")) == NULL) {
      syslog(LOG_ERR,"dd_garbage_d %i: can't open logfile in /tmp\n", pid);
      log_ok=0;
      logfile = stdout;
    }
  }

  if (log_ok) {
    fchmod(fileno(logfile),00777);

    if ( dup2(fileno(logfile),STDERR_FILENO) < 0 ) {
	syslog(LOG_ERR,"dd_garbage_d %i: cannot dup2 the logfile\n", pid);
	exit(1);
    }
    if ( dup2(fileno(logfile),STDOUT_FILENO) < 0 ) {
	syslog(LOG_ERR,"dd_garbage_d %i: cannot dup2 the logfile\n", pid);
	exit(1);
    }
  }

  t = time(NULL);
  fprintf(logfile, "dd_garbage_d: pid=%i, started at %s, attaching\n", pid , ctime(&t));
  dd_nap(0.3);
  dds_create(); 
  fprintf(logfile,"dd_garbage_d %i: dds_created\n", pid);

  strcpy(fname,"GARB");
  for (i=0;i<sizeof(ctl)/4;i++)
	 ctl[i] = 0;
  fmode.mode = FMODE_ALL;
  fmode.wait = FWAIT_SLEEP;
  fmode.prescale = FMODE_NOPRESCALE;
  fmode.suser = FMODE_MULTI_USER;
  fmode.p2ctl = ctl;
  
  if (ddu_init(fname, fmode)) {
    fprintf(logfile,"dd_garbage_d %i: ddu_init failed, exit\n", pid);
    exit(1);
  }

  if (ddu_start()) {
    fprintf(logfile,"dd_garbage_d %i: ddu_start failed, exit\n", pid);
    exit(1);
  }

  dd_dcom->dd_garbage_d_pid = getpid();
  dd_dcom->dd_garbage_d_uid = getuid();

  if (dd_dcom->dd_garbage_d_pid != getpid()) {
    fprintf(logfile,"dd_garbage_d %i: pids are different, exit\n", pid);
    exit(1);
  }

  if (open_status_file() != 0) {
    fprintf(logfile, "dd_garbage_d %i: can't open status file\n", pid);
  }

  error = FALSE;
  fprintf(logfile,"dd_garbage_d %i: up and running\n", pid);
  fflush(logfile);

  signal(DD_SHUTDOWN_SIGNAL, stop_signal);

  start_timer();

  while ( (!error) && (!stop_signal_seen) && (!dd_unused) ) {
    timer_seen = FALSE;
    dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_GETFEV;
    if ( (status = ddu_get_fev(&fev)) == 0 ) {
      if (fev.shmid == -1 ) {
	fprintf(logfile,"dd_garbage_d %i: fev not volatile\n", pid);
	error = TRUE;
      }
      else{
	if (db_detmem(fev.p2da)) {
	  fprintf(logfile,"dd_garbage_d %i: error in db_detmem \n", pid);
	  error = TRUE;
	}
	if (dbfi_free_fev(fev)) {
	  fprintf(logfile,"dd_garbage_d %i: error in dbfi_free_fev \n", pid);
	  error = TRUE;
	}
      }
    }
    else if (status < 0) {
      fprintf(logfile,"dd_garbage_d %i: error in dd_get_fev\n", pid);
      error = TRUE;
    }
    else if (!( (status == EINTR)  && ( stop_signal_seen || timer_seen ) )) {
      fprintf(logfile,"dd_garbage_d %i: dbfi status %i\n", pid, status);
      error = TRUE;
    }
    dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
  }
  
  dd_dcom->dd_garbage_d_pid = -1;
  dd_dcom->dd_garbage_d_uid = -1;
  ddu_stop();
  dds_prepare_shutdown();

/* wait for a bit */
  dd_nap(0.2);
  t = time(NULL);
  
  fprintf(logfile,"dd_garbage_d %i: exit at %s\n", pid, ctime(&t));
  fflush(logfile);

  close_status_file();
  dds_close(1);
  exit(0);
}
