/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *      heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363          *
 *----------------------------------------------------------------------------*
 * Description:
 *	class for general CODA component + tcl init code
 *
 * Author:
 *	Graham Heyes
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *      coda_component.c,v
 *      Revision 2.11  1997/02/11 19:25:00  heyes
 *      cvs stuff
 *
 *      Revision 2.10  1997/02/11 18:59:27  heyes
 *      Add revision info + other things
 *
 *      Revision 2.9  1997/01/16 15:30:37  heyes
 *      Increase speed of EB, inc. changes after Dec run.
 *
 *      Revision 2.8  1996/10/31 15:54:22  abbottd
 *      Changed rocp->active levels for polling, output options
 *
 *      Revision 2.7  1996/10/29 19:39:39  abbottd
 *      Fixed bug in coda_destructor for VxWorks
 *
 *      Revision 2.6  1996/10/29 19:03:11  heyes
 *      new rcServer
 *
 *      Revision 2.5  1996/10/17 14:30:05  heyes
 *      fix EB end problem
 *
 *      Revision 2.4  1996/10/08 17:58:59  heyes
 *      working threaded eb
 *
 *      Revision 2.3  1996/09/19 12:29:47  heyes
 *      Made EB into a real EB
 *
 *      Revision 2.2  1996/09/10 15:44:46  heyes
 *      signal handlers etc
 *
 *      Revision 2.1  1996/09/06 17:37:38  heyes 
 *      Fix error recovery
 *
 *      Revision 2.0  1996/09/06 17:26:01  abbottd
 *      Fixed Automatic  uid and gid detection
 *
 *      Revision 1.7  1996/09/04 13:52:20  heyes
 *      add if around gid code to behave better with old DB format
 *
 *      Revision 1.6  1996/09/04 13:49:09  heyes
 *      add gid and pid support
 *
 *      Revision 1.5  1996/08/29 17:03:56  heyes
 *      move include mempart.h to rc.h
 *
 *      Revision 1.4  1996/08/28 19:10:41  heyes
 *      added support for user selection of data destination.
 *
 *      Revision 1.3  1996/08/28 17:47:28  heyes
 *      removed all ddu_close
 *
 *      Revision 1.2  1996/08/28 17:44:56  heyes
 *      ddu_close
 *
 *      Revision 1.1.1.2  1996/08/22 15:23:01  heyes
 *      Imported sources
 *
 *
 *----------------------------------------------------------------------------*/
/* include files follow here */

#ifdef VXWORKS
 #include "string.h"
 #include <types.h>
 #include <errno.h>
 #include <taskLib.h>
 #include <bootLib.h>
 #include <sysLib.h>
 #include <time.h>
 #include <varargs.h>
#else
 #include <pthread.h>
 #include <sys/resource.h>
 #include <signal.h>
 #include <time.h>
 #include <netinet/in.h>
#endif

#include "da.h"
#include "rc.h"

#ifdef LINUX
 #include <sys/time.h>
 #include <unistd.h>
 #include <stdarg.h>
#endif

#ifdef __sun
 extern char *sys_errlist[];    /* ULTRIX system error decoding... */
#endif

#ifdef WITH_ET
 #include <et_private.h>
 /* more globals */
 static char  *et_filename = NULL;  /* for command line */
 char  et_name[ET_FILENAME_LENGTH];
#endif

/*
 * Start of coda globals etc... There should be no globals !!!!
 */

static char    *msqld_host = NULL;
static char    *msqld_port = NULL;
static char    *session = NULL;
static char    *objects = NULL;
static char    *objectN = NULL;
static char    *objectTy = NULL;

static int      interactive = 0;
static int      runWatchdog = 0;   /* Do not start watchdog by default */
static int      quiet = 1;
static char    *startup_file0 = NULL;
static char    *startup_file = NULL;
static char    *debugString = NULL;
FUNCPTR         process_poll_proc = NULL;
unsigned long *eventNumber;
unsigned long *dataSent;



/*
 * The following variable points to the Tcl interpreter which is used by all
 * functions in this file.
 */
Tcl_Interp * Main_Interp = 0;
/* extern */ int Tk_doneFlag__ = 0;
/*
 * * Static variables used by the interactive Tcl/Tk processing
 */
static Tcl_DString command; /* Used to assemble lines of terminal input into Tcl commands. */

long            rcdebug_level__;
int             global_code[32];
char           *global_routine[8][32];
jmp_buf         global_env[8][32];
long            global_env_depth[32];
long            use_recover = 0;
static long     codaDebugLevel = 0;
int             bigendian = 1;

/* Routine to check the Endianness of host machine */
int isBigEndian(void)
{
  unsigned short num, switched_num;
  
  num = (2<<8) + 1;
  switched_num = htons(num);
  if (num == switched_num)
    return 1;
  else
    return 0;
}

static void codaInitDebug()
{
  char	*env,
  *tmp,
  *tok;

  env = getenv("CODA_DEBUG");

  if(env )
    {
      debugString = (char *)strdup(env);
    }

  if (debugString == NULL)
    return;
  
  printf("\n-------------------------------------------------------\n");
  printf("CODA_DEBUG found. debug started with the following:-\n\n");
  tok = (char *)strtok(debugString,":");
  while(tok)
    {
      if (strcmp(tok,"data") == 0)
	{
	  codaDebugLevel |= 1;
	  printf("Debug level : data transport\n");
	}
      if (strcmp(tok,"api") == 0)
	{
	  codaDebugLevel |= 2;
	  printf("Debug level : api\n");
	}
      if (strcmp(tok,"malloc") == 0)
	{
	  codaDebugLevel |= 4;
	  printf("Debug level : fifo\n");
	}
      tok = (char *)strtok(NULL,":");
    }
  printf("\n-------------------------------------------------------\n\n");
}

void
debug_printf(int level, char *fmt,...)
{
#ifndef VXWORKS68K51
  va_list ap;
  int severity;

#ifdef VXWORKS
  va_start(ap);
#else
  va_start(ap,fmt);
#endif 
 
  if (level & codaDebugLevel) {
    fprintf(stderr, "[CODA debug] ");
    
    vfprintf (stderr,fmt, ap);
  }
#else
   puts("dbg_printf not implemented for 68K 51.");
#endif
}

int pr_time(char *stuff) 
{
  double d1=0.,d2;
#if defined SOLARIS
  struct timespec to;
  clock_gettime(CLOCK_REALTIME,&to);
  d1 = to.tv_sec;
  d2 = to.tv_nsec;
  d1 = d1 + (d2/1000000000.0);
#elif defined LINUX
  struct timeval to;
  gettimeofday(&to,NULL);
  d1 = to.tv_sec;
  d2 = to.tv_usec;
  d1 = d1 + (d2/1000000.0);
#endif
  printf("%f %s\n",d1,stuff);
}
/*
 * C error recovery, we attempt to recover from (normally) fatal system errors.
 */

/**********************************************************
 * System error Recovery - save context to jump to in case
 * of Fatal system error in C code.
 **********************************************************/
 
/*
 * Pop a stored context off the stack (A stack of process contexts is saved, in
 * case of a system level error we can jump back to a safe state. This gives
 * simple error recovery..
 */
int
lastContext ()
{
#define pthread_self() 1
  if (global_env_depth[pthread_self() & 31] > 0)
    global_env_depth[pthread_self() & 31]--;
}

#ifndef VXWORKS

/*
 * Error recovery Init. . .
 */

void *
signal_thread (void *arg)
{
  sigset_t   signal_set;
  int        sig_number, status;
  int        thr = (int) arg;
  char       *rtn;

  if (global_env_depth[thr]>0) {
    global_env_depth[thr]--;
  }
  rtn = global_routine[global_env_depth[thr]][thr];
  
  sigemptyset(&signal_set);
  sigaddset(&signal_set, SIGSEGV);
  sigaddset(&signal_set, SIGBUS);
  sigaddset(&signal_set, SIGILL);
  sigaddset(&signal_set, SIGFPE);
  sigaddset(&signal_set, SIGTRAP);
  sigaddset(&signal_set, SIGINT);
  sigaddset(&signal_set, SIGTERM);
  
  while (1) {
    sigwait(&signal_set, &sig_number);
    switch (sig_number) {
      case SIGSEGV:
        printf ("ERROR Segmentation fault - presumed fatal in %s\n", rtn);
        fprintf (stdout, "ERROR Segmentation fault - presumed fatal in %s\n", rtn);
        break;
    
      case SIGBUS:
        printf ("ERROR Bus error in %s - presumed fatal\n", rtn);
        break;
    
      case SIGILL:
        printf ("ERROR %s, in %s", sys_errlist[errno],rtn);
        perror ("error was-");
        break;
   
      case SIGFPE:
        printf ("ERROR Math error (probably 1/0) in %s\n", rtn);
        break;
    
      case SIGTRAP:
        printf ("ERROR Math error (probably 1/0) in %s\n", rtn);
        break;
         
      case SIGINT:
        printf ("ERROR got SIGINT\n");
	exit(-1);
      case SIGTERM:
        debug_printf (2,"killed by: %s", Tcl_SignalMsg (sig_number));
        Tk_doneFlag__ = 1;
        return;
        break;
        
      default:
        printf ("ERROR Unknown signal %d in %s\n", sig_number, rtn);
        return;
    }
   
    fflush(stdout);
    global_code[thr] = sig_number; 
    siglongjmp(global_env[global_env_depth[thr]][thr],sig_number);
  }
}
#endif

void
Recover_Init ()
{
#ifndef VXWORKS
  pthread_t  id;
  sigset_t   signal_set;
  int        status;
  int        thr = pthread_self() & 31;
  
  sigfillset(&signal_set);
  /*
  sigemptyset(&signal_set);
  sigaddset(&signal_set, SIGSEGV);
  sigaddset(&signal_set, SIGBUS);
  sigaddset(&signal_set, SIGILL);
  sigaddset(&signal_set, SIGFPE);
  sigaddset(&signal_set, SIGTRAP);
  sigaddset(&signal_set, SIGINT);
  sigaddset(&signal_set, SIGTERM);
  */
  status = pthread_sigmask(SIG_BLOCK, &signal_set, NULL);
  if (status != 0) {
    fprintf (stderr, "Recover_Init: error in setting signal mask, %d\n", status);
    exit (3);
  }

  pthread_create(&id, NULL, signal_thread, (void *) thr);
  bzero(global_env_depth, sizeof(global_env_depth));
#endif
}

/*
 * Tcl_AppInit, init all Tcl packages.
 */
int
Tcl_AppInit (interp)
Tcl_Interp     *interp;
{
  /* Tcl Itself */
  if (Tcl_Init (interp) == TCL_ERROR) {
    fprintf (stderr, "Tcl_Init failed: %s\n", interp->result);
    return TCL_ERROR;
  }
  if (Tcl_GlobalEval (interp, "set errorInfo \"not availible\"") != TCL_OK)
    return TCL_ERROR;

  /* DP communications package */
  if (Tdp_Init (interp) == TCL_ERROR) {
    fprintf (stderr, "Tdp_Init failed: %s\n", interp->result);
    return TCL_ERROR;
  }
  /* MSQL database access package */
  if (MSQL_Init (interp) == TCL_ERROR) {
    fprintf (stderr, "MSQL_Init failed: %s\n", interp->result);
    return TCL_ERROR;
  }
  /*Itcl package */
  if (Itcl_Init (interp) == TCL_ERROR) {
    fprintf (stderr, "MSQL_Init failed: %s\n", interp->result);
    return TCL_ERROR;
  }
  if (Struct_Init (interp) == TCL_ERROR) {
    return TCL_ERROR;
  }

  if (LINK_class_Init (interp) == TCL_ERROR) {
    return TCL_ERROR;
  }
  if (Tns_Init (interp) == TCL_ERROR) {
    return TCL_ERROR;
  }
#ifdef TCL_MEM_DEBUG
  Tcl_InitMemory(interp);
#endif
  return TCL_OK;
}

char *user_flag1 = NULL;
char *user_flag2 = NULL;

unsigned long user_flag3,user_flag4;

TCL_PROC (coda_constructor)
{
  Itcl_Object    *obj;


  /* Allocate storage for local class information */
  object = (objClass) ckalloc (sizeof (objClassStore));
  bzero ((char *) object, sizeof (objClassStore));

  /* Get name of this object and extract tail of name */

  if (Tcl_Eval (interp, "set name [info namespace tail $this]") != TCL_OK)
    return TCL_ERROR;

  /* copy name into class structure */

  object->name = (char *) ckalloc (strlen (interp->result) + 1);
  strcpy (object->name, interp->result);

  /* we need to save the pointer to private object storage */

  Tcl_Eval (interp, "set this");				     /* find out who we are */
  Itcl_FindObject (interp, interp->result, &obj);		     /* find pointer to itcl object */

  object->className = obj->cdefn->name;
  object->interp = interp;

  obj->ClientData = (ClientData) object;			     /* store private data */

  if(Tcl_VarEval (interp, "NS_ServerInit ", object->name, " ", session, NULL) != TCL_OK)
    return TCL_ERROR;

  /* tell anyone watching */
  {
    char tmp[400];
    
    sprintf(tmp,"%s {%s} %s {%s}",
	    __FILE__,
	    DAYTIME,
	    CODA_USER,
	    "$Id: coda_component.c,v 2.73 2002/10/21 14:48:42 abbottd Exp $");
    Tcl_SetVar (interp, "tcl_modules",tmp,TCL_LIST_ELEMENT|TCL_APPEND_VALUE|TCL_GLOBAL_ONLY);
  } 
  object->codaid = 0;
  /* link C variables to object */

  eventNumber = (unsigned long *) &object->nevents;
  dataSent = (unsigned long *) &object->nlongs;

  bigendian = isBigEndian(); /* Determine Endianess of Host Machine */

  Tcl_LinkVar (interp, "CODA::state",(char *) &object->state,TCL_LINK_STRING); 
  Tcl_LinkVar (interp, "CODA::name", (char *) &object->name, TCL_LINK_STRING);
  Tcl_LinkVar (interp, "CODA::nlongs", (char *) &object->nlongs, TCL_LINK_INT);
  Tcl_LinkVar (interp, "CODA::nevents", (char *) &object->nevents, TCL_LINK_INT);
  Tcl_LinkVar (interp, "CODA::codaid", (char *) &object->codaid, TCL_LINK_INT);
  Tcl_LinkVar (interp, "CODA::user_flag1", (char *) &user_flag1, TCL_LINK_STRING);
  Tcl_LinkVar (interp, "CODA::user_flag2", (char *) &user_flag2, TCL_LINK_STRING);
  Tcl_LinkVar (interp, "CODA::user_flag3", (char *) &user_flag3, TCL_LINK_INT);
  Tcl_LinkVar (interp, "CODA::user_flag4", (char *) &user_flag4, TCL_LINK_INT);
  Tcl_LinkVar (interp, "CODA::bigendian", (char *) &bigendian, TCL_LINK_INT);

  if (Tcl_Eval (interp, "set user_flag1 NULL;set user_flag2 NULL;set log_name $name") != TCL_OK)
    return TCL_ERROR;

  printf("CODA %s,Name : %s, Type %s Id : %d\x1b[0m\n",
	 VERSION,
	 object->name,
	 object->className,
	 object->codaid);
  
  /* set state to booted */
  if (Tcl_Eval (interp, "status booted") != TCL_OK)
    return TCL_ERROR;

  if (Tcl_Eval (interp, "dp_atexit append \"delete object $this\"") != TCL_OK)
    return TCL_ERROR;

  if (Tcl_VarEval (interp, "every 2000 itcl_context ",object->name," ::CODA setRates",NULL) != TCL_OK)
    return TCL_ERROR;

  if (Tcl_Eval (interp, "every 5000 flush stdout") != TCL_OK)
    return TCL_ERROR;
  if (Tcl_Eval (interp, "every 5000 flush stderr") != TCL_OK)
    return TCL_ERROR;


#ifndef NO_CMLOG
  daLogOpen(object->name,object->className);
  daLogMsg ("INFO", " \"%s\" state now \"%s\"",
	    object->name, object->state);
#endif

  /*
   * Tcl_LinkVar(object->interp, "dp_shmaddr", (char *) &dp_shmaddr,
   * TCL_LINK_INT); Tcl_LinkVar(object->interp, "rcdebug_level", (char *)
   * &rcdebug_level__, TCL_LINK_INT);
   */
 
  if (Tcl_VarEval (interp,
		   "database query \"SELECT id FROM process WHERE name='",
		   object->name,
		   "'\"",
		   NULL) != TCL_OK) {
    return TCL_ERROR;
  }
  if (Tcl_VarEval (interp,
		   "set codaid [database get next]",
		   NULL) != TCL_OK) {
    return TCL_ERROR;
  }
  return TCL_OK;
}

#ifdef NO_CMLOG
void
daLogMsg (char *sev,char *fmt,...)
{
  va_list ap;
  int severity;

  va_start(ap,fmt);
  printf ("severity = %s ",
	  sev);
  vprintf (fmt, ap);
  printf("\n");
  
}
#endif

TCL_PROC(logmsgcmd)
{
  static char *arg1,*arg2;
  arg1 = strdup(argv[1]);
  arg2 = strdup(argv[2]);
  daLogMsg(arg1,arg2);
  ckfree(arg1);
  ckfree(arg2);
  return TCL_OK;
}

TCL_PROC (coda_test)
{
  daLogMsg ("DEBUG", "status of object % is %s", object->name, object->state);
  return TCL_OK;
}

TCL_PROC (coda_destructor)
{

#ifdef ALOG_DEBUG
  ALOG_OUTPUT;
#endif
  
  if ((object != NULL) && (object->name != NULL)) {
    daLogMsg ("WARN", "delete called in object %s", object->name);

    Tcl_UnlinkVar (interp, "CODA::state");
    Tcl_UnlinkVar (interp, "CODA::name");
    Tcl_UnlinkVar (interp, "CODA::nlongs");
    Tcl_UnlinkVar (interp, "CODA::nevents");
    Tcl_UnlinkVar (interp, "CODA::codaid");
    Tcl_UnlinkVar (interp, "CODA::user_flag1");
    Tcl_UnlinkVar (interp, "CODA::user_flag2");
    Tcl_UnlinkVar (interp, "CODA::user_flag3");
    Tcl_UnlinkVar (interp, "CODA::user_flag4");
    Tcl_UnlinkVar (interp, "CODA::bigendian");

    if (Tcl_VarEval (interp,
		     "database query \"UPDATE process SET inuse='no',state='down' WHERE name='",
		     object->name,
		     "'\"",
		     NULL) != TCL_OK) {
      return TCL_ERROR;
    }
  }
  return TCL_OK;
}



/*
 * * Initialize the Tcl/Tk interpreter, plus everything else. *
 * 
 * This is a highly modified version of "main()" from the * file "tkMain.c" in the
 * standard Tcl/Tk distribution.
 */

void
CODA_Init (int argc, char **argv)
{
  Tcl_Interp     *interp; /* Interpreter for this application. */
  char           **listArgv, *args, *obj, *p, buf[20];
  int             code, listArgc;

  /*
   * Static variables used for parsing command line options.
   */
  static Tk_ArgvInfo argTable[] = {
    {"-name", TK_ARGV_STRING, 0, (char *) &objectN,
     "Name of object"},
    {"-type", TK_ARGV_STRING, 0, (char *) &objectTy,
     "Type of object"},
    {"-objects", TK_ARGV_STRING, 0, (char *) &objects,
     "Name and type of this object"},
    {"-msqld_host", TK_ARGV_STRING, 0, (char *) &msqld_host,
     "Name of host to connect to for msql access"},
    {"-msqld_port", TK_ARGV_STRING, 0, (char *) &msqld_port,
     "TCP port to connect to for msql access"},
#ifdef WITH_ET
    {"-et_filename", TK_ARGV_STRING, 0, (char *) &et_filename,
     "Filename of the ET system"},
#endif
    {"-session", TK_ARGV_STRING, 0, (char *) &session,
     "Name of current Session"},
    {"-i", TK_ARGV_CONSTANT, (char *) 1, (char *) &interactive,
     "Interactive mode"},
    {"-w", TK_ARGV_CONSTANT, (char *) 0, (char *) &runWatchdog,
     "Startup Watchdog task"},
    {"-q", TK_ARGV_CONSTANT, (char *) 0, (char *) &quiet,
     "quiet mode"},
    {"-r", TK_ARGV_CONSTANT, (char *) 1, (char *) &use_recover,
     "Error Recover mode"},
    {"-f", TK_ARGV_STRING, (char *) 0, (char *) &startup_file,
     "startup_file"},
    {"-f0", TK_ARGV_STRING, (char *) 0, (char *) &startup_file0,
     "startup_file"},
    {"-debug", TK_ARGV_STRING, (char *) 0, (char *) &debugString,
     "debug mode \"data:api:\""},
    {0, TK_ARGV_END, 0, 0, 0}
  };

  char *argv2[30];
  int argc2;
#ifndef VXWORKS
  {
    int res;
    struct rlimit rls;

#ifndef LINUX    
    res = getrlimit(RLIMIT_VMEM , &rls);
    if (res == 0) {
      debug_printf(2, "CORE limit was %d\n",rls.rlim_cur,rls.rlim_max);
    } else { 
      perror("getrlimit: ");
    }
    res = getrlimit(RLIMIT_AS , &rls);
    if (res == 0) {
      debug_printf(2, "AS limit was %d %d\n",rls.rlim_cur,rls.rlim_max);
    } else { 
      perror("getrlimit: ");
    }
#endif
    res = getrlimit(RLIMIT_DATA , &rls);
    if (res == 0) {
      debug_printf(2, "DATA limit was %d %d\n",rls.rlim_cur,rls.rlim_max);
    } else { 
      perror("getrlimit: ");
    }
    res = getrlimit(RLIMIT_STACK , &rls);
    if (res == 0) {
      debug_printf(2, "STACK limit was %d %d\n",rls.rlim_cur,rls.rlim_max);
    } else { 
      perror("getrlimit: ");
    }
    /*    rls.rlim_cur = 0;
    res = setrlimit(RLIMIT_CORE , &rls);
    if (res == 0) {
      debug_printf(2, "CORE limit set to %d %d\n",rls.rlim_cur);
    } else { 
      perror("getrlimit: ");
    }
    */
  }
#endif /* not VXWORKS */
  argc2 = argc;
  debug_printf(2, "argv0 %s\n",argv[0]);
  {
    int ix;
    for(ix=0;ix<30;ix++) {
      argv2[ix] = argv[ix];
    }
  }
  libPartInit();

  Main_Interp = interp = Tcl_CreateInterp ();

  /*
   * Parse command-line arguments.
   */
parseError:
  if (Tk_ParseArgv (interp, 0, &argc, argv, argTable, 0) != TCL_OK) {
    fprintf (stderr, "%s\n", interp->result);
    exit (3);
  }
  if (msqld_port != NULL) {
    if (Tcl_VarEval (interp, "global env; set env(MSQL_TCP_PORT) ", msqld_port, NULL) != TCL_OK) {
      fprintf (stderr, "%s\n", interp->result);
      exit (3);
    }
  }

  codaInitDebug();
#ifdef VXWORKS
  if (objectN == NULL) {
    objectN = hostname();
  }
#else
  debug_printf(2, "value \"%s\"\n",&argv[0][strlen(argv[0]) - 7]);

  if (!strcmp(&argv[0][strlen(argv[0]) - 7],"coda_eb")) {
    objectTy = strdup("CDEB");
  }
  if (!strcmp(&argv[0][strlen(argv[0]) - 8],"coda_roc")) {
    objectTy = strdup("ROC");
  }
  if (!strcmp(&argv[0][strlen(argv[0]) - 7],"coda_er")) {
    objectTy = strdup("ER");
  }
  if (!strcmp(&argv[0][strlen(argv[0]) - 7],"coda_ts")) {
    objectTy = strdup("TS");
  }

#endif

  if ((objects == NULL) && (objectN !=NULL) && (objectTy !=NULL)) {
    static char temp[100];
    sprintf(temp,"%s %s",objectN, objectTy);
    objects = temp;
  }
  if (session == NULL) {
    session = getenv("SESSION");
  }

  if (objects == NULL || session == NULL) {
    argc = 2;
    argv[1] = "-help";
    goto parseError;
  }
#ifdef NEVER_DEFINED
  {
    char temp[1000];
    int ix;
    sprintf(temp,"EB(%s,%s)",objectN,session);
    argv2[0]=temp;
    execv(argv[0],argv2);
  }
#endif
  {
    static char tmp[300];

    sprintf(tmp,"DD_NAME=%s\0",session);
    putenv(tmp);
  }
    
#ifdef WITH_ET
  if (et_filename == NULL) {
    sprintf(et_name, "%s%s", "/tmp/et_sys_", session);
  }
  else {
    strncpy(et_name, et_filename, ET_FILENAME_LENGTH - 1);
    et_name[ET_FILENAME_LENGTH - 1] = '\0';
  }
#endif

  if (msqld_host) {
    static char tmp[100];
    sprintf(tmp,"MSQL_TCP_HOST=%s",msqld_host);
    debug_printf(2, "MSQL_TCP_HOST=%s\n",msqld_host);
    putenv(tmp);
  } else {
    if (!(msqld_host = getenv("MSQL_TCP_HOST")))
      msqld_host = strdup("localhost");
  }
  /*
   * Make command-line arguments available in the Tcl variables "argc"
   * and "argv".  Also set the "geometry" variable from the geometry
   * specified on the command line.
   */
  args = Tcl_Merge (argc - 1, argv + 1);
  Tcl_SetVar (interp, "argv", args, TCL_GLOBAL_ONLY);
  ckfree (args);
  sprintf (buf, "%d", argc - 1);
  Tcl_SetVar (interp, "argc", buf, TCL_GLOBAL_ONLY);
  Tcl_SetVar (interp, "argv0", argv[0], TCL_GLOBAL_ONLY);

  /*
   * Set the "tcl_interactive" variable.
   */
  if (interactive)
    Tcl_SetVar (interp, "tcl_interactive", "1", TCL_GLOBAL_ONLY);

  Tcl_SetVar (interp, "tcl_modules", "", TCL_GLOBAL_ONLY);
  Tcl_SetVar (interp, "coda_version", VERSION, TCL_GLOBAL_ONLY);
  
  if (Tcl_AppInit (interp)!=TCL_OK) {
    fprintf (stderr, "%s\n", interp->result);
    exit (3);
  }

  /* Set a global Tcl variable here indicating whether
   * the CODA object is running on a Unix or VxWorks platform
   */ 
#ifdef VXWORKS
  Tcl_SetVar(interp,"os_name","vxworks",TCL_GLOBAL_ONLY);
#else
  Tcl_SetVar(interp,"os_name","unix",TCL_GLOBAL_ONLY);
#endif

  Tcl_Eval (interp, "set auto_path \". $auto_path $env(CODA)/common/lib/daq\"");

  if (Tcl_VarEval (interp, "msql connect database ",msqld_host, NULL) != TCL_OK) {
    fprintf (stderr, "%s\n", interp->result);
    exit (3);
  }
  if (Tcl_VarEval (interp, "database set database $env(EXPID)", NULL) != TCL_OK) {
    fprintf (stderr, "%s\n", interp->result);
    exit (3);
  }
  /* It would be nice here to set the uid and gid of the process to 
     match that of whoever is running RunControl */

  {
    static char list[256];
    if (Tcl_VarEval (interp, 
		     "database query \"select owner from sessions where name='",
		     session,
		     "'\"",
		     NULL) != TCL_OK) {
      fprintf (stderr, "%s\n", interp->result);
      fprintf (stderr, "Unable to get owner information\n");
    } else {
      Tcl_Eval(interp, "database get next");
	    
      /* Gid rid of leading and trailing braces */
      if(strlen(interp->result) > 2) {
	strncpy(list,&(interp->result[1]),(strlen(interp->result)-2));
	list[strlen(interp->result)-2] = '\0';
      } else {
	printf("ERROR: No owner information from database table sessions\n");
	printf("string length of interp->result = %d\n",strlen(interp->result));
	exit(3);
      }
      if (Tcl_SplitList (interp, list, &listArgc, &listArgv) != TCL_OK) {
	fprintf (stderr, "%s\n", interp->result);
      } else {
	if(listArgc == 4) {
#ifdef VXWORKS
	  nfsAuthUnixSet(listArgv[0],
			 atoi(listArgv[2]),
			 atoi(listArgv[3]),
			 0,
			 0);
	  nfsAuthUnixShow();
#else
	  /*setuid(atoi(listArgv[2]));
	    setgid(atoi(listArgv[3]));*/
#endif
	} else {
	  printf ("WARNING: Could not get uid and gid info from database\n");
	  printf ("         number of args in the id entry of sessions is %d\n",listArgc);
	}
      }
    }
  }
  
  {
    int             listArgc,
      ix;

    char          **listArgv;

    Itcl_Namespace  ns;

    if (Tcl_SplitList (interp, objects, &listArgc, &listArgv) != TCL_OK) {
      fprintf (stderr, "%s\n", interp->result);
      exit (3);
    }

    if(CODA_class_Init(interp) != TCL_OK) {
      char *val;
      printf("CODA_class_Init");
      fprintf (stderr, "ERROR:\n      %s\n", interp->result);
      val = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
      fprintf (stderr, "TclStack:\n      %s\n",val);
      exit (3);
    }
    
    LINK_support_Init(interp);

    if(LINK_class_Init(interp) != TCL_OK) {
      char *val;
      printf("LINK_class_Init");
      fprintf (stderr, "ERROR:\n      %s\n", interp->result);
      val = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
      fprintf (stderr, "TclStack:\n      %s\n",val);
      exit (3);
    }
    
    if (startup_file0) {
      printf("here with %s\n",startup_file0);
      if (Tcl_EvalFile(interp,(char *) startup_file0) != TCL_OK) {
	fprintf (stderr, "startup file 0 :%s\n", interp->result);
	exit(0);
      }
    }
    for (ix = 0; ix < listArgc; ix += 2) {
      char *class = listArgv[ix+1];
      obj = listArgv[ix];
      if (Tcl_VarEval (interp, class, " ", obj, " ", session, NULL) != TCL_OK) {
	char *val;
	fprintf (stderr, "ERROR:\n      %s\n", interp->result);
	val = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
	fprintf (stderr, "TclStack:\n      %s\n",val);
	exit (3);
      }
    }  
    {
      char tmp[200];
      sprintf(tmp,"puts -nonewline \"%s::%s> \"",session,obj);
      Tcl_SetVar(interp,"tcl_prompt1",tmp, TCL_GLOBAL_ONLY);
    }
    if (interactive) {
      if (!quiet) {
	printf("This is a Tcl shell for control, debugging and access to internal parameters.\n");
	printf("\n");
	printf("For a list of the names of valid commands type...\n\n");
	
	printf("%s info commands\n\n",obj);
      }
      if (!quiet) {
	printf("For a list of configuration parameters type ...\n\n");
	printf("%s configure\n\n",obj);
	printf("Single parameters can be read via ...\n\n");
	printf("%s cget -[name]\n\n",obj);
	printf("...where [name] can be replaced by a name found using \"configure\"\n");
	printf("The versions of C files used to build this code can be found by...\n\n");
	printf("%s version\n\n",obj);
      }
    }

    ckfree ((char *) listArgv);
  }
  /*
   * Error recovery Init. . .
   */
  Recover_Init ();
  if (startup_file)
    Tcl_EvalFile(interp,(char *) startup_file);

  if (interactive && !quiet) {
    printf("Use all other commands with care.\n");
    printf("\n");
  }
}



void
CODA_Service (objClass object)
{
}


/*
 * ----------------------------------------------------------------------
 * 
 * Prompt --
 * 
 * Issue a prompt on standard output, or invoke a script to issue the prompt.
 * 
 * Results: None.
 * 
 * Side effects: A prompt gets output, and a Tcl script may be evaluated in
 * interp.
 * 
 * ----------------------------------------------------------------------
 */
static void
Prompt (interp, partial)
Tcl_Interp     *interp;					     /* Interpreter to use for prompting. */

int             partial;				     /* Non-zero means there already exists a partial command, so
							      * use the secondary prompt. */
{
  char           *promptCmd;

  int             code;

  promptCmd = Tcl_GetVar (interp,
			  partial ? "tcl_prompt2" : "tcl_prompt1", TCL_GLOBAL_ONLY);
  if (promptCmd == NULL) {
  defaultPrompt:
    if (!partial) {
      fputs ("% ", stdout);
    }
  } else {
    code = Tcl_Eval (interp, promptCmd);
    if (code != TCL_OK) {
      Tcl_AddErrorInfo (interp,
			"\n    (script that generates prompt)");
      fprintf (stderr, "%s\n", interp->result);
      goto defaultPrompt;
    }
  }
  fflush (stdout);
}

/*
 * ----------------------------------------------------------------------
 * 
 * StdinProc --
 * 
 * This procedure is invoked by the event dispatcher whenever standard input
 * becomes readable.  It grabs the next line of input characters, adds them to
 * a command being assembled, and executes the command if it's complete.
 * 
 * Results: None.
 * 
 * Side effects: Could be almost arbitrary, depending on the command that's typed.
 * 
 * ----------------------------------------------------------------------
 */
static int oldStdin, oldStdout, oldStderr;

/* ARGSUSED */
static void
StdinProc (clientData, mask)
ClientData      clientData;				     /* Not used. */

int             mask;					     /* Not used. */
{
  char            input[4000];

  static int      gotPartial = 0;

  char           *cmd;

  int             code,
    count;
 
  count = read (0, input, sizeof (input) - 1);
  if (count <= 0) {
    if (!gotPartial) {
#ifdef VXWORKS
      int fd;
      char cmd[200];
      fd = ioTaskStdGet(0,0);
      printf("close connection file%d\n",fd);
      if (!interactive)
	Tk_DeleteFileHandler (0);
      sprintf(cmd,"close file%d\n",fd);
      Tcl_Eval(Main_Interp,cmd);
      ioTaskStdSet(0,0,oldStdin);
      ioTaskStdSet(0,1,oldStdout);
      ioTaskStdSet(0,2,oldStderr);
      return;
#else
	Tcl_Eval (Main_Interp, "exit");
	exit (6);
#endif
    } else {
      count = 0;
    }
  }
  cmd = Tcl_DStringAppend (&command, input, count);

  if (count != 0) {
    if ((input[count - 1] != '\n') && (input[count - 1] != ';')) {
      gotPartial = 1;
      goto prompt;
    }
    if (!Tcl_CommandComplete (cmd)) {
      gotPartial = 1;
      goto prompt;
    }
  }
  gotPartial = 0;

  /*
   * Disable the stdin file handler while evaluating the command;
   * otherwise if the command re-enters the event loop we might process
   * commands from stdin before the current command is finished.  Among
   * other things, this will trash the text of the command being
   * evaluated.
   */

  Tk_CreateFileHandler (0, 0, StdinProc, (ClientData) 0);
  code = Tcl_RecordAndEval (Main_Interp, cmd, 0);
  Tk_CreateFileHandler (0, TK_READABLE, StdinProc, (ClientData) 0);
  Tcl_DStringFree (&command);
  if (*Main_Interp->result != 0) {
    printf ("%s\n", Main_Interp->result);
  }

  /*
   * Output a prompt.
   */
prompt:
  Prompt (Main_Interp, gotPartial);
}

/*
 * Main program loop...
 */
#ifdef VXWORKS
static int watchFlag;

void codaWatchdog()
{
  while (1) {

    watchFlag = 1;
    taskDelay (60*sysClkRateGet()); /* ten second timeout */

    if (watchFlag) {
      int i;
      for(i=8;i<32;i++) 
	close(i);
      break;
    }
  }
  printf("codaWatchdog done\n");
}


static void watching ()
{
  watchFlag = 0;
  Tk_CreateTimerHandler (2000, (Tk_TimerProc *) watching, (ClientData) NULL);
}
#endif

TCL_PROC(CODAConnectStdio)
{
#ifdef VXWORKS
  FILE           *filePtr;
  int fd;

  oldStdin  = ioTaskStdGet(0,0);
  oldStdout = ioTaskStdGet(0,1);
  oldStderr = ioTaskStdGet(0,2);
  if (Tcl_GetOpenFile (interp, argv[1], 1, 1, &filePtr) != TCL_OK) {
    return TCL_ERROR;
  }
  fd = fileno (filePtr);

  ioTaskStdSet(0,0,fd);
  ioTaskStdSet(0,1,fd);
  ioTaskStdSet(0,2,fd);

  if (!interactive)
    Tk_CreateFileHandler (0, TK_READABLE, StdinProc, (ClientData) 0);

#endif
  return TCL_OK;
}

TCL_PROC(CODADisconnectStdio)
{
#ifdef VXWORKS
  ioTaskStdSet(0,0,oldStdin);
  ioTaskStdSet(0,1,oldStdout);
  ioTaskStdSet(0,2,oldStderr);
#endif
  return TCL_OK;
}

void
CODA_Execute ()
{
  
  int fd;
  int status;
  
  /*
   * If the following lines are removed then tcl_interactive must be set
   * to 0
   */
#ifdef VXWORKS
    Tk_CreateTimerHandler (2000, (Tk_TimerProc *) watching, (ClientData) NULL);

    /* wait for shell to start */
    do
      taskDelay (sysClkRateGet ());
    while (taskNameToId ("tShell") == ERROR);

    if (runWatchdog) {
      printf("\nStart watchdog\n");

      sp(codaWatchdog);
    }
#endif
  if (interactive) {
    
#ifdef VXWORKS
    printf("\nThis is VxWorks so disallow Tcl exit\n");
    Tcl_Eval(Main_Interp,"proc exit {} {puts \"exit not allowed\"}");
    printf("Readout controller ready to roll...\n");
#endif
    Tk_CreateFileHandler (0, TK_READABLE, StdinProc, (ClientData) 0);

    Prompt (Main_Interp, 0);
  }

  Tcl_DStringInit (&command);
  
  /*
   * Process Tk events until the last window is destroyed, then die
   * gracefully.  This function never returns.
   */
  {
    int res;
    recoverContext ("main program loop",res);
#ifndef VXWORKS
    while (!Tk_doneFlag__) {
#else
    while (1) {
#endif
      (void)Tk_DoOneEvent(0);
    }
      
#ifdef VXWORKS
    Tcl_DStringFree(&command);
    Tcl_DeleteInterp(Main_Interp);
#else
    Tcl_Eval(Main_Interp, "exit");
    exit(1);
#endif
    exit (5);
    }
    
  }
  
/*
 *-------------------------------------------------------------------------*
 *   Routine to take a Data buffer in CODA format and perform a
 *   a byte swap based on the data type in the Bank structure headers.
 *
 *   Data is maniputlated in the existing buffer so that function
 *   irreversably mangles the data. If there is an error in the Bank
 *   structure the function returns -1, otherwize it returns 0;
 *   
 *-------------------------------------------------------------------------*
 */
int
CODA_bswap(cbuf, nlongs)
  long *cbuf;
  int nlongs;
{
    int ii, jj, ix;
    int tlen, blen, dtype;
    long lwd;
    short shd;
    char cd;
    char *cp;
    short *sp;
    long *lp;

    ii = 0;
    while (ii<nlongs) {
      lp = (long *)&cbuf[ii];
      blen = cbuf[ii] - 1;
      dtype = ((cbuf[ii+1])&0xff00)>>8;
      lwd = LSWAP(*lp);    /* Swap the length      */
      *lp++ = lwd;         
      lwd = LSWAP(*lp);    /* Swap the bank header */
      *lp++ = lwd;       
      ii += 2;
      if(dtype != DT_BANK) {
	switch(dtswap[dtype]) {
	case 0:
	  /* No swap */
	  ii += blen;
	  break;
	case 1:
	  /* short swap */
	  sp = (short *)&cbuf[ii];
	  for(jj=0; jj<(blen<<1); jj++) {
	    shd = SSWAP(*sp);
	    *sp++ = shd;
	  }
	  ii += blen;
	  break;
	case 2:
          /* long swap */
	  lp = (long *)&cbuf[ii];
	  for(jj=0; jj<blen; jj++) {
	    lwd = LSWAP(*lp);
	    *lp++ = lwd;
	  }
	  ii += blen;
	  break;
	case 3:
	  /* double swap */
	  lp = (long *)&cbuf[ii];
	  for(jj=0; jj<blen; jj++) {
	    lwd = LSWAP(*lp);
	    *lp++ = lwd;
	  }
	  ii += blen;
	  break;
	default:
	  /* No swap */
	  ii += blen;
	}
      }
    }
 return(0);
}
