/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *      heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363          *
 *----------------------------------------------------------------------------*
 * Discription: follows this header.
 *
 * Author:
 *	Graham Heyes
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *      $Log: BOS_format.c,v $
 *      Revision 2.4  1998/11/13 14:40:01  abbottd
 *      Updated event type mask from 16 bits to 8 bits
 *
 *      Revision 2.3  1998/05/27 13:44:55  heyes
 *      add message Q to ROC, improve EB stability on EBD transition
 *
 *      Revision 2.1  1996/11/01 13:39:13  heyes
 *      initial revision of event recorder
 *
 *      Revision 2.3  1996/10/29 19:03:09  heyes
 *      new rcServer
 *
 *      Revision 2.2  1996/09/19 17:16:06  heyes
 *      Support for BOS file format
 *
 *      Revision 2.1  1996/09/19 14:14:08  heyes
 *      Initial release
 *
 *      Revision 1.2  1996/08/29 17:03:58  heyes
 *----------------------------------------------------------------------------*/
#define ERROR -1
#define OK 0
#define NOT_TYPE 1

#include "BOS_format.h"

static unsigned long *frag_start;
static unsigned long *head_start;
static unsigned long *desc_start;

int 
BOS_reserv_frag(unsigned long **datap)
{
  /* CODA headers are two long words */

  frag_start = *datap;
  *datap +=2;
}

int 
BOS_reserv_head(unsigned long **datap)
{
  /* CODA headers are two long words */

  head_start = *datap;
  *datap +=2;
}

int 
BOS_reserv_desc(unsigned long **datap)
{
  /* CODA ID banks are five long words */

  desc_start = *datap;
  *datap +=5;
}

int 
BOS_decode_frag(unsigned long **datap, evDesc desc)
{
  desc->length = ((*datap)[0] - 1) << 2;
  desc->evnb   = (*datap)[1] & 0xff;
  desc->type   = ((*datap)[1] >> 16) & 0x00ff;
  desc->fragments[0] = &(*datap)[2];
  desc->bankTag[0] = desc->rocid;
  desc->fragLen[0] = desc->length;
  desc->bankCount = 1;
  desc->soe = *datap;

  /* CODA headers are two long words */
  *datap += 2;

  /* fragment headers have a variable format and are hard to 
     check, we assume here that if the length is resonable 
     then so is the rest of the event. NOTE: limit 0.5MBy.
     */
  if (desc->length < 500000) {
    return OK;
  } else {
    return ERROR;
  }
}

int 
BOS_encode_frag(unsigned long **datap, evDesc desc)
{
  /* CODA event lengths are "non inclusive" i.e. don't include
     the length itself */

  frag_start[0] = *datap -frag_start -1;
  frag_start[1] = ((desc->rocid & 0xffff) << 16) | (desc->evnb & 0xff) | 0x0100;
}

int 
BOS_decode_head(unsigned long **datap, evDesc desc)
{
  unsigned long marker;

  /* The only usefull data here are the total event length and type */

  desc->type   = ((*datap)[1] >> 16 ) & 0x00ff;

  desc->length = ((*datap)[0] - 1) << 2;
  marker = (*datap)[1] & 0xffff;

  /* CODA headers are two long words */

  *datap += 2;

  /* CODA event header must have 0x10CC as low 16-bits */

  if ( marker == 0x10CC ) {
    return OK;
  } else {
    return ERROR;
  }
}

int 
BOS_encode_head(unsigned long **datap, evDesc desc)
{
  /* CODA event lengths are "non inclusive" i.e. don't include
     the length itself */

  head_start[0] = *datap - head_start -1 ;
  head_start[1] = 0x10CC | ((desc->type & 0x00ff) << 16);
  desc->length = (*datap - head_start)<<2;
}

int 
BOS_encode_desc(unsigned long **datap, evDesc desc)
{
  /* CODA event lengths are "non inclusive" i.e. don't include
     the length itself */

  desc_start[0] = 4;
  desc_start[1] = 0xC0000100;
  desc_start[2] = desc->evnb;
  desc_start[3] = desc->type;
  desc_start[4] = desc->err[1];
}

int 
BOS_decode_desc(unsigned long **datap, evDesc desc)
{
  unsigned long marker;

  /* CODA event lengths are "non inclusive" i.e. don't include
     the length itself */

  marker = (*datap)[1];
  desc->evnb = (*datap)[2];
 
  desc->type = (*datap)[3];
  desc->err[1] = (*datap)[4];

  *datap += 5;

  if ( marker == 0xC0000100 ) {
    return OK;
  } else {
    return ERROR;
  }  
}

int 
BOS_decode_spec(unsigned long **datap, evDesc desc)
{
  unsigned long marker;

  desc->length = ((*datap)[0] - 1) << 2;

  desc->type  = ((*datap)[1] >> 16) & 0xffff;

  marker = (*datap)[1] & 0xffff;

  desc->time = (*datap)[2];

  switch(desc->type == 16) {
  case 16:
    {
      desc->syncev = (*datap)[3];
      desc->evnb = (*datap)[4];
      desc->err[1] = (*datap)[5];
    } 
  case 17: 
    {
      desc->runnb = (*datap)[3];
      desc->runty = (*datap)[4];
    }
  default: 
    {
      desc->evnb = (*datap)[4];
    }
  }
  *datap += ((*datap)[0] + 1);

  if ( marker == 0x01CC ) {
    return OK;
  } else {
    return ERROR;
  }  
}

int 
BOS_encode_spec(unsigned long **datap, evDesc desc)
{
  (*datap)[1] = 0x01CC | (desc->type << 16);

  (*datap)[2] = desc->time;

  switch(desc->type == 16) {
  case 16:
    {
    (*datap)[0] = 5;
    (*datap)[3] = desc->syncev;
    (*datap)[4] = desc->evnb;
    (*datap)[5] = desc->err[1];
    } 
  case 17: 
    {
    (*datap)[0] = 4;
    (*datap)[3] = desc->runnb;
    (*datap)[4] = desc->runty;
    }
  default: 
    {
    (*datap)[0] = 4;
    (*datap)[3] = 0;
    (*datap)[4] = desc->evnb;
    }
  }

  desc->length = ((*datap)[0] + 1)<<2;
  *datap += ((*datap)[0] + 1);
}


