/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *       heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363         *
 *----------------------------------------------------------------------------*
 * Description: follows this header.
 *
 * Author:
 *	David Abbott
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *	  Initial revision
 *
 *
 *----------------------------------------------------------------------------*/

 /* sfi_trigLib.c -- Primitive trigger control for the SFI 340 using available
                     interrupt sources including CEBAF designed AUX
                     Trigger supervisor interface card.
                            intmask -- refers to the bit pattern one wishes
                                       to use to enable certain trigger sources
                                       (see the VME IRQ Source and Mask Register)
                                            NIM1,NIM2,NIM3
                                            ECL1
                                            AUX_B42
                                            etc...

 File : sfi_trigLib.h

 Supported Interrupt/Trigger sources:

       code       Hardware type
        1     SFI Trigger Supervisor AUX Interface

 Routines:
           void sfiAuxWrite();       write 32bit word to AUX port
           void sfiAuxRead();        read 32bit word from AUX port
	   void sfitriglink();       link in trigger isr
	   void sfitenable();        enable trigger
	   void sfitdisable();       disable trigger
           void sfitack();           Acknowledge/Reset trigger
	   char sfittype();          read trigger type
	   int  sfittest();          test for trigger (POLL)

------------------------------------------------------------------------------*/

/* Define Hardware sources */
#define AUX_SOURCE 1

/* Define Interrupt Vector and VME level */
#define SFI_VEC   0xec
#define SFI_LEVEL 5

#ifdef VXWORKSPPC
#define INUM_TO_IVEC(intNum)    ((VOIDFUNCPTR *) (intNum))
#else
#define INUM_TO_IVEC(intNum)    ((VOIDFUNCPTR *) (intNum<<2))
#endif

/* Define global variable for AUX board revision */
unsigned int sfiAuxVersion = 0; /* Default is an invalid Version ID */

#define SFI_AUX_VERSION_MASK  0x00006000 
#define SFI_AUX_VERSION_1     0x00006000
#define SFI_AUX_VERSION_2     0x00004000


void
sfiAuxWrite(val32)
     unsigned long val32;
{
  *sfi.sequencerDisable = 0;
  *sfi.writeAuxReg = val32;
  *sfi.writeVmeOutSignalReg = 0x4000;
  *sfi.generateAuxB40Pulse = 0;
  *sfi.writeVmeOutSignalReg = 0x40000000;
}

unsigned long
sfiAuxRead()
{
  int xx=0;
  unsigned long val32 = 0xffffffff;
  *sfi.sequencerDisable = 0;
  *sfi.writeVmeOutSignalReg = 0x2000;
  while ( (xx<10) && (val32 == 0xffffffff) ) {
    val32 = *sfi.readLocalFbAdBus;
    xx++;
  }
  *sfi.writeVmeOutSignalReg = 0x20000000;
  return (val32);
}

int
sfiAuxInit()
{
  unsigned long rval = 0;

  /* Check if SFI AUX board is readable */
  rval = sfiAuxRead();
  if (rval == 0xffffffff) {
    printf("sfiAuxInit: ERROR: AUX card not addressable\n");
    return(-1);
  } else {
    sfiAuxWrite(0x8000); /* Reset the board */
    sfiAuxVersion = (SFI_AUX_VERSION_MASK & rval);
    printf("sfiAuxInit: sfiAuxVersion = 0x%x\n",sfiAuxVersion);
  }

  return(0);
  
}

void 
sfitriglink(int code, VOIDFUNCPTR isr)
{

/* Enable  SFI to Inturrupt VME bus */  
    *sfi.VmeIrqLevelAndVectorReg = (0x800 | (SFI_LEVEL<<8) | SFI_VEC);


#ifdef VXWORKSPPC
  if((intDisconnect(INUM_TO_IVEC(SFI_VEC)) !=0))
     printf("Error disconnecting Interrupt\n");
#endif
  intConnect(INUM_TO_IVEC(SFI_VEC),isr,0);

}

void 
sfitenable(int code, unsigned int intMask)
{
 int lock_key;

#ifndef POLLING_MODE
  lock_key = intLock();

  *sfi.VmeIrqMaskReg  = (intMask<<8);         /* Reset any Pending Trigger */ 
  *sfi.VmeIrqMaskReg  = intMask;              /* Enable source */
  sysIntEnable(SFI_LEVEL);                    /* Enable VME Level 5 interrupts */
#ifdef VXWORKSPPC
  intEnable(11);       /* Enable VME Interrupts on IBC chip */
#endif

#ifdef TRIG_SUPV
/*Set AUX Port for Trigger Supervisor triggers*/
  sfiAuxWrite(0);
#else
/*Set AUX Port for External triggers*/
  if(intMask == 0x10) {
    if(sfiAuxVersion == SFI_AUX_VERSION_2)
      sfiAuxWrite(0x300);
    else
      sfiAuxWrite(0x180);
  }
#endif
  intUnlock(lock_key);

#else

  *sfi.VmeIrqLevelAndVectorReg = ( (SFI_LEVEL<<8) | SFI_VEC);     /* Enable Internal Interrupts */
  *sfi.VmeIrqMaskReg  = (intMask<<8);                             /* Reset any Pending Trigger */ 
  *sfi.VmeIrqMaskReg  = intMask;                                  /* Enable source*/

#ifdef TRIG_SUPV
/*Set AUX Port for Trigger Supervisor triggers*/
  sfiAuxWrite(0);
#else
/*Set AUX Port for External triggers*/
  if(intMask == 0x10) {
    if(sfiAuxVersion == SFI_AUX_VERSION_2)
      sfiAuxWrite(0x300);
    else
      sfiAuxWrite(0x180);
  }
#endif

#endif
}

void 
sfitdisable(int code, unsigned int intMask)
{

  *sfi.VmeIrqMaskReg  = (intMask<<8);      /* Clear Source  */
#ifndef POLLING
  sysIntDisable(SFI_LEVEL);        /* Disable VME Level 5 interrupts */
#endif

/*Reset AUX Port */
  if(intMask == 0x10) {
    sfiAuxWrite(0x8000);
  }
}

void 
sfitack(int code, unsigned int intMask)
{
#ifdef POLLING_MODE
  *sfi.VmeIrqMaskReg  = (intMask<<8);      /* Clear Source  */
#endif

  if(intMask == 0x10) {
    *sfi.writeVmeOutSignalReg = 0x1000;       /* Set A10 Ack       */
    *sfi.VmeIrqMaskReg  = intMask;            /* Enable Source  */
    *sfi.writeVmeOutSignalReg = 0x10000000;   /* Clear A10 Ack     */
  } else {
    *sfi.VmeIrqMaskReg  = intMask;            /* Enable Source  */
  }
}

unsigned long 
sfittype(int code)
{
  unsigned long tt;

#ifdef TRIG_SUPV
/* Read Trigger type from TS AUX port */
  sfiAuxPort = sfiAuxRead();
  if(sfiAuxPort == 0xffffffff)
    logMsg("sfittype: ERROR: Error reading Aux port\n",0,0,0,0,0,0);

  if(sfiAuxVersion == SFI_AUX_VERSION_2)
    tt       = ((sfiAuxPort&0xfc)>>2);
  else
    tt       = ((sfiAuxPort&0x3c)>>2);

  lateFail = ((sfiAuxPort&0x02)>>1);
  syncFlag = ((sfiAuxPort&0x01));

#else
/* Set default trigger type to 1 */
  tt=1;
#endif
  return(tt);
}

int 
sfittest(int code)
{
  int ii=0;
  /* NOTE: See that internal VME IRQ is set on the SFI*/
  ii = (((*sfi.VmeIrqLevelAndVectorReg) & 0x4000) != 0);
  return(ii);
}

