/******************************************************************************
*
*  fadcLib.h  - Driver library header file for readout of the JLAB 250MHz FLASH
*                ADC using a VxWorks 5.5 or later based single board 
*                computer.
*
*  Author: David Abbott 
*          Jefferson Lab Data Acquisition Group
*          June 2007
*
*  Revision  1.0 - Initial Revision
*                    - Supports up to 20 FADC boards in a Crate
*                    - Programmed I/O and Block reads
*
*  Revision  1.1 - Updated for new firmware (0x0111 or later)
*                  Supports FADC Signal Distribution Module
*
*/

#define FA_BOARD_ID       0xfadc0000
#define FA_MAX_BOARDS             20
#define FA_MAX_ADC_CHANNELS       16
#define FA_MAX_DATA_PER_CHANNEL  251
#define FA_MAX_A32_MEM      0x800000   /* 8 Meg */
#define FA_MAX_A32MB_SIZE   0x800000  /*  8 MB */
#define FA_VME_INT_LEVEL           3     
#define FA_VME_INT_VEC          0xFA


struct fadc_struct {
  volatile unsigned int version;		/* (0x000 - 0x0FC) - CTRL */
  volatile unsigned int csr;
  volatile unsigned int ctrl1;
  volatile unsigned int ctrl2;
  volatile unsigned int blk_level;
  volatile unsigned int intr;
  volatile unsigned int adr32;
  volatile unsigned int adr_mb;
  volatile unsigned int s_adr;
  volatile unsigned int delay;
  volatile unsigned int itrig_cfg;
  volatile unsigned int reset;
  volatile unsigned int trig_scal;
  volatile unsigned int ev_count;
  volatile unsigned int blk_count;
  volatile unsigned int blk_fifo_count;
  volatile unsigned int blk_wrdcnt_fifo;
  volatile unsigned int internal_trig_scal;
  volatile unsigned int ram_word_count;
  volatile unsigned int dataflow_status;
  volatile unsigned short dac[16];
  volatile unsigned int status[4];
  volatile unsigned int aux[4];
  volatile unsigned int ram_r_adr;
  volatile unsigned int ram_w_adr;
  volatile unsigned int ramL_data;
  volatile unsigned int ramH_data;
  volatile unsigned int berr_module_scal;
  volatile unsigned int berr_crate_scal;
  volatile unsigned int aux_scal[6];
  volatile unsigned int busy_level;
  volatile unsigned int moller_scal_status;
  volatile unsigned int moller_scal_fifo;
  volatile unsigned int trigger_data_status;
  volatile unsigned int itrig_burst_count;
  volatile unsigned int itrig_burst_ctrl;
  volatile unsigned int spare_ctrl[10];
  
  volatile unsigned short adc_status[2];	/* (0x100 - 0x1FC) - ADC processing */
  volatile unsigned short adc_status1[2];
  volatile unsigned short adc_cfg[2];
  volatile unsigned short adc_ptw[2];
  volatile unsigned short adc_pl[2];
  volatile unsigned short adc_nsb[2];
  volatile unsigned short adc_nsa[2];
  volatile unsigned short adc_spare[2];
  volatile unsigned short adc_thres[16];
  volatile unsigned short ptw_last_adr[2];
  volatile unsigned short ptw_max_buf[2];
  volatile unsigned int spare_adc[46];

#ifdef INCLUDE_MOLLER  
  volatile unsigned int moller_status[2];	/* (0x200 - 0x2FC) - MOLLER processing */
  volatile unsigned int reserve_scal1[10];
  volatile unsigned int CL_sample_cnt;
  volatile unsigned int CL_sum_thresh[2];
  volatile unsigned int CR_sample_cnt;
  volatile unsigned int CR_sum_thresh[2];
  volatile unsigned int CL_pulse_width;
  volatile unsigned int CR_pulse_width;
  volatile unsigned int SL_pulse_width[4];
  volatile unsigned int SR_pulse_width[4];
  volatile unsigned int CL_delay;
  volatile unsigned int CR_delay;
  volatile unsigned int SL_delay[4];
  volatile unsigned int SR_delay[4];
  volatile unsigned int SLSR_delay;
  volatile unsigned int CR_trig_prescale;
  volatile unsigned int CL_trig_prescale;
  volatile unsigned int CLCR_trig_prescale;
  volatile unsigned int moller_trig_fifo;
  volatile unsigned int moller_trig_window;
  volatile unsigned int reserve_scal2[4];
  volatile unsigned int spare_hitsum[16];
#else
  volatile unsigned int hitsum_status;	        /* (0x200 - 0x2FC) - HITSUM processing */
  volatile unsigned int hitsum_cfg;
  volatile unsigned int hitsum_width;	        /*  secondary address (0x0 - 0xF) */
  volatile unsigned int hitsum_delay;
  volatile unsigned int hitsum_trig_width;
  volatile unsigned int hitsum_trig_bits;
  volatile unsigned int hitsum_window_width;
  volatile unsigned int hitsum_overlap_bits;
  volatile unsigned int hitsum_pattern;	         /*  secondary address (0x0 - 0x1000) */
  volatile unsigned int hitsum_fifo;
  volatile unsigned int hitsum_sum_thres;
  volatile unsigned int spare_hitsum[52];
#endif
};

struct fadc_data_struct {
  unsigned int new_type;	
  unsigned int type;	
  unsigned int slot_id_hd;
  unsigned int slot_id_tr;
  unsigned int n_evts;
  unsigned int blk_num;
  unsigned int n_words;
  unsigned int evt_num_1;
  unsigned int evt_num_2;
  unsigned int time_now;
  unsigned int time_1;
  unsigned int time_2;
  unsigned int time_3;
  unsigned int time_4;
  unsigned int chan;
  unsigned int width;
  unsigned int valid_1;
  unsigned int adc_1;
  unsigned int valid_2;
  unsigned int adc_2;
  unsigned int over;
  unsigned int adc_sum;
  unsigned int pulse_num;
  unsigned int thres_bin;
  unsigned int quality;
  unsigned int integral;
  unsigned int time;
  unsigned int chan_a;
  unsigned int source_a;
  unsigned int chan_b;
  unsigned int source_b;
  unsigned int group;
  unsigned long time_coarse;
  unsigned long time_fine;
  unsigned long vmin;
  unsigned long vpeak;
  unsigned int trig_type_int;	/* next 4 for internal trigger data */
  unsigned int trig_state_int;	/* e.g. helicity */
  unsigned int evt_num_int;
  unsigned int err_status_int;
};


struct fadc_sdc_struct {
  volatile unsigned short csr;
  volatile unsigned short ctrl;
  volatile unsigned short busy_enable;
  volatile unsigned short busy_status;
};


/* FADC Special Board IDs */
#define  FA_MOLLER_BOARD_ID        0xfadc00a0
#define  FA_MOLLER_BOARD_MASK      0xffff00f0


/* Define CSR Bits */
#define  FA_CSR_EVENT_AVAILABLE           0x1
#define  FA_CSR_BLOCK_LEVEL_FLAG          0x2
#define  FA_CSR_BLOCK_READY               0x4
#define  FA_CSR_BERR_STATUS               0x8
#define  FA_CSR_TOKEN_STATUS             0x10
#define  FA_CSR_PLL1_LOCKED              0x20
#define  FA_CSR_PLL2_LOCKED              0x40
#define  FA_CSR_PLL3_LOCKED              0x80
#define  FA_CSR_PLL_MASK                 0xe0
#define  FA_CSR_PLL1_LOCK_LOST          0x100
#define  FA_CSR_PLL2_LOCK_LOST          0x200
#define  FA_CSR_PLL3_LOCK_LOST          0x400
#define  FA_CSR_PLL_LOCK_LOST_MASK      0x700
#define  FA_CSR_FIFO1_EMPTY             0x800
#define  FA_CSR_FIFO1_ALMOST_EMPTY     0x1000
#define  FA_CSR_FIFO1_HALF_FULL        0x2000
#define  FA_CSR_FIFO1_ALMOST_FULL      0x4000
#define  FA_CSR_FIFO1_FULL             0x8000
#define  FA_CSR_FIFO2_EMPTY           0x10000
#define  FA_CSR_FIFO2_ALMOST_EMPTY    0x20000
#define  FA_CSR_FIFO2_HALF_FULL       0x40000
#define  FA_CSR_FIFO2_ALMOST_FULL     0x80000
#define  FA_CSR_FIFO2_FULL           0x100000
#define  FA_CSR_HSFIFO_EMPTY         0x200000
#define  FA_CSR_HSFIFO_ALMOST_EMPTY  0x400000
#define  FA_CSR_HSFIFO_HALF_FULL     0x800000
#define  FA_CSR_HSFIFO_ALMOST_FULL  0x1000000
#define  FA_CSR_HSFIFO_FULL         0x2000000
#define  FA_CSR_ERROR_MASK         0x0c000000
#define  FA_CSR_ERROR_CLEAR        0x08000000
#define  FA_CSR_SYNC               0x10000000
#define  FA_CSR_TRIGGER            0x20000000
#define  FA_CSR_SOFT_RESET         0x40000000
#define  FA_CSR_HARD_RESET         0x80000000

/* Define Init Flag bits for Clock Source */
#define FA_SOURCE_INT         0x00
#define FA_SOURCE_SDC         0x10
#define FA_SOURCE_VXS         0x20
#define FA_SOURCE_BDC         0x30
#define FA_SOURCE_MASK        0x30

/* Define Control2 Bits */
#define FA_CTRL_GO               0x1
#define FA_CTRL_ENABLE_TRIG      0x2
#define FA_CTRL_ENABLE_SRESET    0x4
#define FA_CTRL_ENABLE_INT_TRIG  0x8
#define FA_CTRL_ENABLE_MASK      0x7
#define FA_CTRL_ENABLED          0x7
/* Define Moller Control2 Bits */
#define FA_MOLLER_CTRL_ENABLE_MASK    0xb
#define FA_MOLLER_CTRL_ENABLED        0xb


/* Define CTRL1 Bits */
#define FA_REF_CLK_P2              0x0
#define FA_REF_CLK_FP              0x1
#define FA_REF_CLK_P0              0x5
#define FA_REF_CLK_INTERNAL        0x7
#define FA_REF_CLK_MASK            0x7
#define FA_ENABLE_INTERNAL_CLK     0x8

#define FA_TRIG_FP                0x00
#define FA_TRIG_FP_ISYNC          0x10
#define FA_TRIG_P0                0x20
#define FA_TRIG_P0_ISYNC          0x30
#define FA_TRIG_P2                0x40
#define FA_TRIG_P2_ISYNC          0x50
#define FA_TRIG_VME               0x60
#define FA_TRIG_INTERNAL          0x70
#define FA_TRIG_MASK              0x70
#define FA_ENABLE_SOFT_TRIG       0x80

#define FA_SRESET_FP             0x000
#define FA_SRESET_FP_ISYNC       0x100
#define FA_SRESET_P0             0x200
#define FA_SRESET_P0_ISYNC       0x300
#define FA_SRESET_P2             0x400
#define FA_SRESET_P2_ISYNC       0x500
#define FA_SRESET_VME            0x600
#define FA_SRESET_NONE           0x700
#define FA_SRESET_MASK           0x700
#define FA_ENABLE_SOFT_SRESET    0x800

#define FA_ENABLE_LIVE_TRIG_OUT  0x1000
#define FA_ENABLE_TRIG_OUT_FP    0x2000
#define FA_ENABLE_TRIG_OUT_P0    0x4000
#define FA_ENABLE_TRIG_OUT_P2    0x8000

#define FA_ENABLE_CHAN_0_7        0x10000
#define FA_ENABLE_CHAN_8_15       0x20000
#define FA_ENABLE_BLKLVL_INT      0x40000
#define FA_ENABLE_ERROR_INT       0x80000
#define FA_ENABLE_BERR           0x100000
#define FA_ENABLE_MULTIBLOCK     0x200000
#define FA_FIRST_BOARD           0x400000
#define FA_LAST_BOARD            0x800000
#define FA_BYPASS_EXT_RAM       0x1000000
#define FA_ENABLE_DEBUG         0x2000000
#define FA_MB_TOKEN_VIA_P0     0x10000000
#define FA_MB_TOKEN_VIA_P2     0x20000000

/* Define Control 2 Bits */
#define FA_GO                         0x1
#define FA_ENABLE_EXT_TRIGGER         0x2
#define FA_ENABLE_EXT_SRESET          0x4
#define FA_ENABLE_INT_TRIGGER         0x8
#define FA_ENABLE_STREAM_MODE        0x10
#define FA_SELECT_STREAM_DATA_8_15   0x20

#define FA_DEBUG_XFER_INFIFO_BFIFO       0x1000        
#define FA_DEBUG_XFER_BFIFO_OFIFO        0x2000        
#define FA_DEBUG_DISABLE_XFIFO0      0x40000000        
#define FA_DEBUG_DISABLE_XFIFO1      0x80000000        

/* Define Reset Bits */
#define FA_RESET_CNTL_FPGA         0x1
#define FA_RESET_ADC_FPGA1         0x2
#define FA_RESET_ADC_FPGA2         0x4
#define FA_RESET_HITSUM_FPGA       0x8
#define FA_RESET_ADC_FIFO1       0x100
#define FA_RESET_ADC_FIFO2       0x200
#define FA_RESET_HITSUM_FIFO     0x400
#define FA_RESET_DAC             0x800
#define FA_RESET_TOKEN         0x10000
#define FA_RESET_ITRIG_BURST   0x20000

/* Define RAM Bits */
#define FA_RAM_DATA_MASK       0x000fffff
#define FA_RAM_FULL            0x00100000
#define FA_RAM_EMPTY           0x00200000
#define FA_RAM_ADR_INCREMENT   0x00100000

/* Define Bit Masks */
#define FA_VERSION_MASK        0x000000ff
#define FA_BOARD_MASK          0xffff0000
#define FA_CSR_MASK            0x0fffffff
#define FA_CONTROL_MASK        0xffffffff
#define FA_CONTROL2_MASK       0xc000303f
#define FA_EVENT_COUNT_MASK    0xffffff
#define FA_BLOCK_COUNT_MASK    0xfffff
#define FA_BLOCK_LEVEL_MASK    0xffff
#define FA_INT_ENABLE_MASK     0xc0000
#define FA_INT_VEC_MASK        0xff
#define FA_INT_LEVEL_MASK      0x700
#define FA_SLOT_ID_MASK        0x1f0000
#define FA_DAC_VALUE_MASK      0x00ff
#define FA_DAC_LOAD            0x8000

#define FA_REF_CLK_SEL_MASK        0x7
#define FA_TRIG_SEL_MASK          0x70
#define FA_SRESET_SEL_MASK       0x700

#define FA_BUSY_LEVEL_MASK     0xfffff
#define FA_FORCE_BUSY       0x80000000

#define FA_A32_ENABLE        0x1
#define FA_AMB_ENABLE        0x1
#define FA_A32_ADDR_MASK     0xff80   /* 8 MB chunks */
#define FA_AMB_MIN_MASK      0xff80
#define FA_AMB_MAX_MASK      0xff800000

/* Define default ADC Processing prarmeters 
   values are in clock cycles 4ns/cycle */

#define FA_ADC_NS_PER_CLK      4

#define FA_ADC_PROC_MASK        0x7
#define FA_ADC_PROC_ENABLE      0x8
#define FA_ADC_PROC_MODE_WINDOW   1
#define FA_ADC_PROC_MODE_PEAK     2
#define FA_ADC_PROC_MODE_SUM      3
#define FA_ADC_PROC_MODE_TIME     7

#define FA_ADC_VERSION_MASK  0x7fff
#define FA_ADC_BIT_MASK      0x8000
#define FA_ADC_NBITS_10      0x8000
#define FA_ADC_NBITS_12      0x0000

#define FA_ADC_PEAK_MASK     0x0070
#define FA_ADC_CHAN_MASK     0xff00

#define FA_ADC_DEFAULT_PL     50
#define FA_ADC_DEFAULT_PTW    50
#define FA_ADC_DEFAULT_NSB     5
#define FA_ADC_DEFAULT_NSA    10
#define FA_ADC_DEFAULT_NP      4
#define FA_ADC_DEFAULT_THRESH  0

#define FA_ADC_MAX_PL       1000
#define FA_ADC_MAX_PTW       500
#define FA_ADC_MAX_NSB       500
#define FA_ADC_MAX_NSA       500
#define FA_ADC_MAX_NP          1
#define FA_ADC_MAX_THRESH   1023

#define FA_ADC_MIN_PL          1
#define FA_ADC_MIN_PTW         1
#define FA_ADC_MIN_NSB         1
#define FA_ADC_MIN_NSA         1
#define FA_ADC_MIN_NP          1
#define FA_ADC_MIN_THRESH      0


/* Define ADC Data Types and Masks */

#define FA_DATA_TYPE_DEFINE       0x80000000
#define FA_DATA_TYPE_MASK         0x78000000
#define FA_DATA_SLOT_MASK         0x07c00000

#define FA_DATA_BLOCK_HEADER      0x00000000
#define FA_DATA_BLOCK_TRAILER     0x08000000
#define FA_DATA_EVENT_HEADER      0x10000000
#define FA_DATA_TRIGGER_TIME      0x18000000
#define FA_DATA_WINDOW_RAW        0x20000000
#define FA_DATA_WINDOW_SUM        0x28000000
#define FA_DATA_PULSE_RAW         0x30000000
#define FA_DATA_PULSE_INTEGRAL    0x38000000
#define FA_DATA_PULSE_TIME        0x40000000
#define FA_DATA_STREAM            0x48000000
#define FA_DATA_INVALID           0x70000000
#define FA_DATA_FILLER            0x78000000
#define FA_DUMMY_DATA             0xf800fafa

#define FA_DATA_BLKNUM_MASK       0x0000003f
#define FA_DATA_WRDCNT_MASK       0x003fffff
#define FA_DATA_TRIGNUM_MASK      0x07ffffff


/* Define FADC Signal Distribution card bits */

#define FA_SDC_BOARD_ID    0x0200
#define FA_SDC_BOARD_MASK  0xfffe
#define FA_SDC_ADR_MASK    0xffc0

#define FASDC_BUSY_MASK   0x00ff

#define FASDC_CSR_BUSY           0x1
#define FASDC_CSR_TRIG          0x20
#define FASDC_CSR_SRESET        0x40
#define FASDC_CSR_INIT          0x80
#define FASDC_CSR_MASK        0x00e1

#define FASDC_CTRL_CLK_EXT               0x1
#define FASDC_CTRL_NOSYNC_TRIG          0x10
#define FASDC_CTRL_ENABLE_SOFT_TRIG     0x20
#define FASDC_CTRL_NOSYNC_SRESET       0x100
#define FASDC_CTRL_ENABLE_SOFT_SRESET  0x200
#define FASDC_CTRL_ASSERT_SOFT_BUSY   0x8000
#define FASDC_CTRL_MASK               0x8331



/* Function Prototypes */
STATUS faInit (UINT32 addr, UINT32 addr_inc, int nadc, int iFlag);
void faStatus(int id, int sflag);
int faSetProcMode(int id, int pmode, unsigned int PL, unsigned int PTW, 
	          unsigned int NSB, unsigned int NSA, unsigned int NP, int bank);
int faItrigBurstConfig(int id, unsigned int ntrig, 
		   unsigned int burst_window, unsigned int busy_period);
unsigned int faItrigControl(int id, unsigned short itrig_width, unsigned short itrig_dt);
int faReadBlock(int id, volatile UINT32 *data, int nwrds, int rflag);
int faPrintBlock(int id, int rflag);
void faClear(int id);
void faClearError(int id);
unsigned int faReadCSR(int id);
void faReset(int id, int iFlag);
void faResetToken(int id);
void faChanDisable(int id, unsigned short cmask);
void faEnable(int id, int eflag, int bank);
void faDisable(int id, int eflag);
void faTrig(int id);
void faSync(int id);
int faDready(int id,int dflag);
int faBready(int id);
int faGetRes(int id);
int faBusyLevel(int id, unsigned int val, int bflag);
int faBusy(int id);
void faEnableSoftTrig(int id);
void faDisableSoftTrig(int id);
void faEnableSoftSync(int id);
void faDisableSoftSync(int id);
void faEnableClk(int id);
void faDisableClk(int id);
void faEnableBusError(int id);
void faDisableBusError(int id);
void faEnableMultiBlock(int tflag);
void faDisableMultiBlock();
int faSetBlockLevel(int id, int level);
int faSetClkSource(int id, int source);
int faSetTrigSource(int id, int source);
int faSetSyncSource(int id, int source);
void faEnableFP(int id);
int faSetThreshold(int id, unsigned short tvalue, unsigned short chmask);
int faPrintThreshold(int id);
int faSetDAC(int id, unsigned short dvalue, unsigned short chmask);
void faPrintDAC(int id);
void faDataDecode(unsigned int data);

/* FLASH SDC prototypes */
int faSDC_Config(unsigned short cFlag, unsigned short bMask);
void faSDC_Status(int sFlag);
void faSDC_Enable(int nsync);
void faSDC_Disable();
void faSDC_Sync();
void faSDC_Trig();
int faSDC_Busy();

/* Moller Extension Prototypes */
#ifdef INCLUDE_MOLLER
int famInit (UINT32 addr, int iFlag);
int famSetSumThresh(int det, unsigned int lr, unsigned int val);
int famSetSampleCount(int det, unsigned int lr, unsigned int val);
int famSetPrescale(int det, unsigned int lr, unsigned int val);
int famSetPulseWidth(int det, unsigned int lr, unsigned int val);
int famSetDelay(int det, unsigned int lr, unsigned int val);
int famSetTrigWindow(unsigned int val);
void famEnable();
void famDisable(int dFlag);
void famEnableTest();
void famDisableTest(int dFlag);
void famReset();
void famMollerDataAck();
int famReadScalers(volatile unsigned int *data);
unsigned int famScalStatus(int sFlag);
int faMollerStatus(int sFlag);
#endif
