/********************************************************************************
*  fadcLib.c  -  Driver library for JLAB config and readout of JLAB 250MHz FLASH
*                  ADC using a VxWorks 5.4 or later based Single Board computer. 
*
*  Author: David Abbott 
*          Jefferson Lab Data Acquisition Group
*          June 2007
*
*  Revision  1.0 - Initial Revision
*                    - Supports up to 20 FADC boards in a Crate
*                    - Programmed I/O and Block reads
*
*  Revision  1.1 - Updated for new firmware (0x0111 or later)
*                  Supports FADC Signal Distribution Module
*
*
*/

#include <vxWorks.h>
#include <stdio.h>
#include <string.h>
#include <logLib.h>
#include <taskLib.h>
#include <intLib.h>
#include <iv.h>
#include <semLib.h>
#include <vxLib.h>


/* Include ADC definitions */
#include "fadcLib.h"

/* Include DMA Library definitions */


/* Define external Functions */
IMPORT  STATUS sysBusToLocalAdrs(int, char *, char **);
IMPORT  STATUS intDisconnect(int);
IMPORT  STATUS sysIntEnable(int);
IMPORT  STATUS sysIntDisable(int);

#define EIEIO    __asm__ volatile ("eieio")
#define SYNC     __asm__ volatile ("sync")

/* Define Interrupts variables */
BOOL              fadcIntRunning  = FALSE;                    /* running flag */
int               fadcIntID       = -1;                       /* id number of ADC generating interrupts */
LOCAL VOIDFUNCPTR fadcIntRoutine  = NULL;                     /* user interrupt service routine */
LOCAL int         fadcIntArg      = 0;                        /* arg to user routine */
LOCAL UINT32      fadcIntLevel    = FA_VME_INT_LEVEL;         /* default VME interrupt level */
LOCAL UINT32      fadcIntVec      = FA_VME_INT_VEC;           /* default interrupt Vector */

/* Define global variables */
int nfadc = 0;                                       /* Number of FADCs in Crate */
int fadcA32Base   = 0x08000000;                      /* Minimum VME A32 Address for use by FADCs */
int fadcA32Offset = 0x08000000;                      /* Difference in CPU A32 Base - VME A32 Base */
volatile struct fadc_struct *FAp[(FA_MAX_BOARDS+1)]; /* pointers to FADC memory map */
volatile struct fadc_sdc_struct *FASDCp;             /* pointer to FADC Signal distribution card */
volatile unsigned int *FApd[(FA_MAX_BOARDS+1)];      /* pointers to FADC FIFO memory */
volatile unsigned int *FApmb;                        /* pointer to Multblock window */
int fadcID[FA_MAX_BOARDS];                           /* array of slot numbers for FADCs */
int fadcRev[(FA_MAX_BOARDS+1)];                      /* Board Revision Info for each module */
unsigned short fadcChanDisable[(FA_MAX_BOARDS+1)];   /* Disabled Channel Mask for each Module*/
int fadcInited=0;                                    /* >0 if Library has been Initialized before */
int fadcMaxSlot=0;                                   /* Highest Slot hold an FADC */
int fadcMinSlot=0;                                   /* Lowest Slot holding an FADC */
int fadcSource=0;                                    /* Signal source for FADC system control*/
int fadcBlockLevel=0;                                /* Block Level for ADCs */
int fadcIntCount = 0;                                /* Count of interrupts from FADC */
int fadcUseSDC=0;                                    /* If > 0 then Use Signal Distribution board */
struct fadc_data_struct fadc_data;


/* Include Moller extensions if necessary */
#ifdef INCLUDE_MOLLER
#include "mollerLib.c"
#endif


/*******************************************************************************
*
* faInit - Initialize JLAB FADC Library. 
*
*
*   iFlag: 16 bit integer
*       Low 6 bits - Specifies the default Signal distribution (clock,trigger) 
*                    sources for the board (INTernal, FrontPanel, VXS, VME(Soft))
*       bit    0:  defines Sync Reset source
*                     0  VME (Software Sync-Reset)
*                     1  Front Panel/VXS/P2 (Depends on Clk/Trig source selection)
*       bits 3-1:  defines Trigger source
*               0 0 0  VME (Software Triggers)
*               0 0 1  Front Panel Input
*               0 1 0  VXS (P0) 
*               0 1 1  P2 Connector (Backplane)
*               1 0 0  Internal Trigger Logic (HITSUM FPGA)
*               (all others Undefined - default to Internal)
*       bits 5-4:  defines Clock Source
*           0 0  Internal 250MHz Clock
*           0 1  Front Panel 
*           1 0  VXS (P0)
*           1 1  P2 Connector (Blackplane)
*
*       Common Modes of Operation:
*           Value = 0  CLK (Int)  TRIG (Soft)   SYNC (Soft)    (Debug/Test Mode)
*                   2  CLK (Int)  TRIG (FP)     SYNC (Soft)    (Single Board
*                   3  CLK (Int)  TRIG (FP)     SYNC (FP)         Modes)
*                0x10  CLK (FP)   TRIG (Soft)   SYNC (Soft)
*                0x13  CLK (FP)   TRIG (FP)     SYNC (FP)      (VME SDC Mode)
*                0x20  CLK (VXS)  TRIG (Soft)   SYNC (Soft)
*                0x25  CLK (VXS)  TRIG (VXS)    SYNC (VXS)     (VXS SD Mode)
*
*
*      High 10bits - A16 Base address of FADC Signal Distribution Module
*                    This board can control up to 7 FADC Boards.
*                    Clock Source must be set to Front Panel (bit4 = 1)
*
*
* RETURNS: OK, or ERROR if the address is invalid or a board is not present.
*/

STATUS 
faInit (UINT32 addr, UINT32 addr_inc, int nadc, int iFlag)
{
  int ii, res, rdata, errFlag = 0;
  int boardID = 0;
  int maxSlot = 1;
  int minSlot = 21;
  int trigSrc, clkSrc, srSrc;
  unsigned int laddr, a32addr, a16addr=0;
  volatile struct fadc_struct *fa;
  unsigned short saddr, sdata;


  /* Check if we have already Initialized boards before */
  if((fadcInited>0) && (fadcID[0] != 0)) {
    /* Hard Reset of all FADC boards in the Crate */
    for(ii=0;ii<nfadc;ii++) {
      FAp[fadcID[ii]]->csr  = FA_CSR_HARD_RESET;
    }
    taskDelay(5);
  }
  
  /* Check for valid address */
  if(addr==0) {
    printf("faInit: ERROR: Must specify a Bus (VME-based A24) address for FADC 0\n");
    return(ERROR);
  }else if(addr > 0x00ffffff) { /* A24 Addressing */
    printf("faInit: ERROR: A32 Addressing not allowed for FADC configuration space\n");
    return(ERROR);
  }else{ /* A24 Addressing */
    if((addr_inc==0)||(nadc==0))
      nadc = 1; /* assume only one FADC to initialize */

    /* get the FADC address */
    res = sysBusToLocalAdrs(0x39,(char *)addr,(char **)&laddr);
    if (res != 0) {
      printf("faInit: ERROR in sysBusToLocalAdrs(0x39,0x%x,&laddr) \n",addr);
      return(ERROR);
    }
  }

  /* Init Some Global variables */
  fadcSource = iFlag&FA_SOURCE_MASK;
  fadcInited = nfadc = 0;
  fadcUseSDC = 0;
  bzero((char *)fadcChanDisable,sizeof(fadcChanDisable));
  bzero((char *)fadcID,sizeof(fadcID));

  for (ii=0;ii<nadc;ii++) {
    fa = (struct fadc_struct *)(laddr + ii*addr_inc);
    /* Check if Board exists at that address */
    res = vxMemProbe((char *) &(fa->version),VX_READ,4,(char *)&rdata);
    if(res < 0) {
      printf("faInit: ERROR: No addressable board at addr=0x%x\n",(UINT32) fa);
      errFlag = 1;
      break;
    } else {
      /* Check that it is an FA board */
      if((rdata&FA_BOARD_MASK) != FA_BOARD_ID) {
	printf(" ERROR: Invalid Board ID: 0x%x\n",rdata);
        return(ERROR);
      }
      /* Check if this is board has a valid slot number */
      boardID =  ((fa->intr)&FA_SLOT_ID_MASK)>>16;
      if((boardID <= 0)||(boardID >21)) {
        printf(" ERROR: Board Slot ID is not in range: %d\n",boardID);
        return(ERROR);
      }
      FAp[boardID] = (struct fadc_struct *)(laddr + ii*addr_inc);
      fadcRev[boardID] = rdata&FA_VERSION_MASK;
    }
    fadcID[nfadc] = boardID;
    if(boardID >= maxSlot) maxSlot = boardID;
    if(boardID <= minSlot) minSlot = boardID;

    nfadc++;
    printf("Initialized FADC %d  Slot # %d at address 0x%08x \n",ii,fadcID[ii],(UINT32) FAp[(fadcID[ii])]);
  }

  /* Check if we are using a JLAB FADC Signal Distribution Card (SDC)
     NOTE the SDC board only supports 7 FADCs - so if there are
     more than 7 FADCs in the crate they can only be controlled by daisychaining 
     multiple SDCs together - or by using a VXS Crate with SD switch card 
  */
  a16addr = iFlag&FA_SDC_ADR_MASK;
  if(a16addr) {
    res = sysBusToLocalAdrs(0x29,(char *)a16addr,(char **)&laddr);
    if (res != 0) {
      printf("faInit: ERROR in sysBusToLocalAdrs(0x29,0x%x,&laddr) \n",a16addr);
      return(ERROR);
    }

    res = vxMemProbe((char *) laddr,VX_READ,2,(char *)&sdata);
    if(res < 0) {
      printf("faInit: ERROR: No addressable SDC board at addr=0x%x\n",(UINT32) laddr);
    } else {
      FASDCp = (struct fadc_sdc_struct *) laddr;
      FASDCp->ctrl = FASDC_CSR_INIT;   /* Reset the Module */

      if(nfadc>7) {
	printf("WARN: A Single JLAB FADC Signal Distribution Module only supports 7 FADCs\n");
	printf("WARN: You must use multiple SDCs to support more FADCs - this must be configured in hardware\n");
      }
      printf("Using JLAB FADC Signal Distribution Module at address 0x%x\n",
	       (UINT32) FASDCp); 
      fadcUseSDC=1;
    }
    if(fadcSource == FA_SOURCE_SDC) {  /* Check if SDC will be used */
      fadcUseSDC = 1;
      printf("faInit: JLAB FADC Signal Distribution Card is Assumed in Use\n");
      printf("faInit: Front Panel Inputs will be enabled. \n");
    }else{
      fadcUseSDC = 0;
      printf("faInit: JLAB FADC Signal Distribution Card will not be Used\n");
    }
  }

  /* Hard Reset of all FADC boards in the Crate */
  for(ii=0;ii<nfadc;ii++) {
    FAp[fadcID[ii]]->csr  = FA_CSR_HARD_RESET;
  }
  taskDelay(5);

  /* Initialize Interrupt variables */
  fadcIntID = -1;
  fadcIntRunning = FALSE;
  fadcIntLevel = FA_VME_INT_LEVEL;
  fadcIntVec = FA_VME_INT_VEC;
  fadcIntRoutine = NULL;
  fadcIntArg = 0;

  /* Calculate the A32 Offset for use in Block Transfers */
    res = sysBusToLocalAdrs(0x09,(char *)fadcA32Base,(char **)&laddr);
    if (res != 0) {
      printf("faInit: ERROR in sysBusToLocalAdrs(0x09,0x%x,&laddr) \n",fadcA32Base);
      return(ERROR);
    } else {
      fadcA32Offset = laddr - fadcA32Base;
    }


  /* what are the Trigger Sync Reset and Clock sources */
    if ((fadcSource == FA_SOURCE_VXS)||(fadcSource == FA_SOURCE_BDC)) {
      printf("faInit: Enabling FADC for VXS Clock ");
      clkSrc  = FA_REF_CLK_P0;
      switch (iFlag&0xf) {
      case 0: case 1:
	printf("and Software Triggers (Soft Sync Reset)\n");
	trigSrc = FA_TRIG_VME | FA_ENABLE_SOFT_TRIG;
	srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	break;
      case 2:
	printf("and Front Panel Triggers (Soft Sync Reset)\n");
	trigSrc = FA_TRIG_FP_ISYNC;
	srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	break;
      case 3:
	printf("and Front Panel Triggers (FP Sync Reset)\n");
	trigSrc = FA_TRIG_FP_ISYNC;
	srSrc   = FA_SRESET_FP_ISYNC;
	break;
      case 4: case 6:
	printf("and VXS Triggers (Soft Sync Reset\n");
	trigSrc = FA_TRIG_P0_ISYNC;
	srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	break;
      case 5: case 7:
	printf("and VXS Triggers (VXS Sync Reset\n");
	trigSrc = FA_TRIG_P0_ISYNC;
	srSrc   = FA_SRESET_P0_ISYNC;
	break;
      case 8: case 10: case 12: case 14:
	printf("and Internal Trigger Logic (Soft Sync Reset)\n");
	trigSrc = FA_TRIG_INTERNAL;
	srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	break;
      case 9: case 11: case 13: case 15:
	printf("and Internal Trigger Logic (VXS Sync Reset)\n");
	trigSrc = FA_TRIG_INTERNAL;
	srSrc   = FA_SRESET_FP_ISYNC;
	break;
      }
    }else if (fadcSource == FA_SOURCE_SDC) {
      printf("faInit: Enabling FADC for SDC Clock (Front Panel) ");
      clkSrc  = FA_REF_CLK_FP;
      switch (iFlag&0xf) {
      case 0: case 1:
	printf("and Software Triggers (Soft Sync Reset)\n");
	trigSrc = FA_TRIG_VME | FA_ENABLE_SOFT_TRIG;
	srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	break;
      case 2: case 4: case 6:
	printf("and Front Panel Triggers (Soft Sync Reset)\n");
	trigSrc = FA_TRIG_FP_ISYNC;
	srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	break;
      case 3: case 5: case 7:
	printf("and Front Panel Triggers (FP Sync Reset)\n");
	trigSrc = FA_TRIG_FP_ISYNC;
	srSrc   = FA_SRESET_FP_ISYNC;
	break;
      case 8: case 10: case 12: case 14:
	printf("and Internal Trigger Logic (Soft Sync Reset)\n");
	trigSrc = FA_TRIG_INTERNAL;
	srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	break;
      case 9: case 11: case 13: case 15:
	printf("and Internal Trigger Logic (Front Panel Sync Reset)\n");
	trigSrc = FA_TRIG_INTERNAL;
	srSrc   = FA_SRESET_FP_ISYNC;
	break;
      }
      faSDC_Config(0,0);
    }else {  /* Use internal Clk */
       printf("faInit: Enabling FADC Internal Clock, ");
       clkSrc = FA_REF_CLK_INTERNAL;
       switch (iFlag&0xf) {
       case 0: case 1:
	 printf("and Software Triggers (Soft Sync Reset)\n");
	 trigSrc = FA_TRIG_VME | FA_ENABLE_SOFT_TRIG;
	 srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET ;
 	 break;
       case 2:
	 printf("and Front Panel Triggers (Soft Sync Reset)\n");
	 trigSrc = FA_TRIG_FP_ISYNC;
	 srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	 break;
       case 3:
	 printf("and Front Panel Triggers (FP Sync Reset)\n");
	 trigSrc = FA_TRIG_FP_ISYNC;
	 srSrc   = FA_SRESET_FP_ISYNC;
	 break;
       case 4: case 6:
	 printf("and VXS Triggers (Soft Sync Reset\n");
	 trigSrc = FA_TRIG_P0_ISYNC;
	 srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	 break;
       case 5: case 7:
	 printf("and VXS Triggers (VXS Sync Reset\n");
	 trigSrc = FA_TRIG_P0_ISYNC;
	 srSrc   = FA_SRESET_P0_ISYNC;
	 break;
       case 8: case 10: case 12: case 14:
	 printf("and Internal Trigger Logic (Soft Sync Reset)\n");
	 trigSrc = FA_TRIG_INTERNAL;
	 srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;
	 break;
       case 9: case 11: case 13: case 15:
	 printf("and Internal Trigger Logic (Front Panel Sync Reset)\n");
	 trigSrc = FA_TRIG_INTERNAL;
	 srSrc   = FA_SRESET_FP_ISYNC;
	 break;
       }
    }


    /* Enable Clock source - Internal Clk enabled by default */ 
  for(ii=0;ii<nfadc;ii++) {
    FAp[fadcID[ii]]->ctrl1 = (clkSrc | FA_ENABLE_INTERNAL_CLK) ;
  }
  taskDelay(20);

  for(ii=0;ii<nfadc;ii++) {
    FAp[fadcID[ii]]->reset = (FA_RESET_ADC_FPGA1 | FA_RESET_ADC_FPGA2 | FA_RESET_HITSUM_FPGA |
                              FA_RESET_ADC_FIFO1 | FA_RESET_ADC_FIFO2 | FA_RESET_HITSUM_FIFO);
  }
  taskDelay(5);

  /* Write configuration registers with default/defined Sources */
  for(ii=0;ii<nfadc;ii++) {
    
    /* Program an A32 access address for this FADC's FIFO */
    a32addr = fadcA32Base + ii*FA_MAX_A32_MEM;
    res = sysBusToLocalAdrs(0x09,(char *)a32addr,(char **)&laddr);
    if (res != 0) {
      printf("faInit: ERROR in sysBusToLocalAdrs(0x09,0x%x,&laddr) \n",a32addr);
      return(ERROR);
    }
    FAp[fadcID[ii]]->adr32 = (a32addr>>16) + 1;  /* Write the register and enable */
    FApd[fadcID[ii]] = (unsigned int *)(laddr);  /* Set a pointer to the FIFO */


    /* Set Default Block Level to 1 */
    FAp[fadcID[ii]]->blk_level = 1;
    fadcBlockLevel=1;

    /* Setup Trigger and Sync Reset sources */
    FAp[fadcID[ii]]->ctrl1 |= (srSrc | trigSrc | FA_ENABLE_CHAN_0_7 | FA_ENABLE_CHAN_8_15);
  }

  /* If there are more than 1 FADC in the crate then setup the Muliblock Address
     window. This must be the same on each board in the crate */
  if(nfadc > 1) {
    a32addr = fadcA32Base + (nfadc+1)*FA_MAX_A32_MEM; /* set MB base above individual board base */
    res = sysBusToLocalAdrs(0x09,(char *)a32addr,(char **)&laddr);
    if (res != 0) {
      printf("faInit: ERROR in sysBusToLocalAdrs(0x09,0x%x,&laddr) \n",a32addr);
      return(ERROR);
    }
    FApmb = (unsigned int *)(laddr);  /* Set a pointer to the FIFO */
    for (ii=0;ii<nfadc;ii++) {
      /* Write the register and enable */
      FAp[fadcID[ii]]->adr_mb = (a32addr+FA_MAX_A32MB_SIZE) + (a32addr>>16) + FA_A32_ENABLE;
    }
    
    /* Set First Board and Last Board */
    fadcMaxSlot = maxSlot;
    fadcMinSlot = minSlot;
    FAp[minSlot]->ctrl1 |= FA_FIRST_BOARD;
    FAp[maxSlot]->ctrl1 |= FA_LAST_BOARD;
    
  }

  fadcInited = nfadc;
  if(errFlag > 0) {
    printf("faInit: ERROR: Unable to initialize all FADC Modules\n");
    if(nfadc > 0)
      printf("faInit: %d FADC(s) successfully initialized\n",nfadc );
    return(ERROR);
  } else {
    return(OK);
  }
}


void
faStatus(int id, int sflag)
{
  int ii, jj;
  unsigned int a32Base, ambMin, ambMax, vers, bid, brev;
  unsigned int csr, ctrl1, ctrl2, count, bcount, blevel, intr, addr32, addrMB;
  unsigned short adcStat0[2], adcStat1[2], adcConf[2], 
                 PTW[2], PL[2], NSB[2], NSA[2], NP, adcChanDisabled;
  unsigned int adc_enabled, adc_10bit, adc_version, adc_option;
  unsigned int trigCnt, itrigCnt, ramWords, moller;
  unsigned short sdc[4];

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    printf("faStatus: ERROR : ADC in slot %d is not initialized \n",id);
    return;
  }

  vers   =  FAp[id]->version;
  bid    = ((vers)&FA_BOARD_MASK)>>16;
  brev   = (vers)&FA_VERSION_MASK;

/* Check if this is a Moller Board Revision */
  if(((vers)&FA_MOLLER_BOARD_MASK) == FA_MOLLER_BOARD_ID)
    moller = 1;
  else
    moller = 0;

  csr    = (FAp[id]->csr)&FA_CSR_MASK;
  ctrl1  = (FAp[id]->ctrl1)&FA_CONTROL_MASK;
  ctrl2  = (FAp[id]->ctrl2)&FA_CONTROL2_MASK;
  count  = (FAp[id]->ev_count)&FA_EVENT_COUNT_MASK;
  bcount = (FAp[id]->blk_count)&FA_BLOCK_COUNT_MASK;
  blevel  = (FAp[id]->blk_level)&FA_BLOCK_LEVEL_MASK;
  ramWords = (FAp[id]->ram_word_count)&FA_RAM_DATA_MASK;
  trigCnt = FAp[id]->trig_scal;
  itrigCnt = FAp[id]->internal_trig_scal;
  intr   = FAp[id]->intr;
  addr32 = FAp[id]->adr32;
  a32Base = (addr32&FA_A32_ADDR_MASK)<<16;
  addrMB = FAp[id]->adr_mb;
  ambMin =  (addrMB&FA_AMB_MIN_MASK)<<16;
  ambMax =  (addrMB&FA_AMB_MAX_MASK);

  /* If moller=1 and the DAQ is active then we cant access the FPGAs */
  if((moller)&&(ctrl2&FA_CTRL_ENABLE_TRIG)) {
    moller = 2;
  }else{
    for(ii=0;ii<2;ii++) {
      adcStat0[ii] = FAp[id]->adc_status[ii];
      adcStat1[ii] = FAp[id]->adc_status1[ii];
      adcConf[ii] = FAp[id]->adc_cfg[ii];
      PTW[ii] =  FAp[id]->adc_ptw[ii]*FA_ADC_NS_PER_CLK;
      PL[ii] =  FAp[id]->adc_pl[ii]*FA_ADC_NS_PER_CLK;
      NSB[ii] =  FAp[id]->adc_nsb[ii]*FA_ADC_NS_PER_CLK;
      NSA[ii] =  FAp[id]->adc_nsa[ii]*FA_ADC_NS_PER_CLK;
    }
    adc_10bit   = adcStat1[0]&FA_ADC_BIT_MASK;
    adc_version = adcStat1[0]&FA_ADC_VERSION_MASK;
    adc_option  = (adcConf[0]&FA_ADC_PROC_MASK) + 1;
    NP          = (adcConf[0]&FA_ADC_PEAK_MASK)>>4;
    adc_enabled = (adcConf[0]&FA_ADC_PROC_ENABLE);
    adcChanDisabled = (adcConf[1]&FA_ADC_CHAN_MASK) | ((adcConf[0]&FA_ADC_CHAN_MASK)>>8);
  }


  printf("\nSTATUS for FADC in slot %d at base address 0x%x \n",id,(UINT32) FAp[id]);
  printf("-------------------------------------------------- \n");

  printf(" Board Firmware Rev/ID = 0x%04x : ADC Processing Rev = 0x%04x\n",
	 (vers)&0xffff, adc_version);
  if(addrMB&FA_AMB_ENABLE) {
    printf(" Alternate VME Addressing: Multiblock Enabled\n");
    if(addr32&FA_A32_ENABLE)
      printf("   A32 Enabled at VME base 0x%08x or CPU addr 0x%08x\n",a32Base,(UINT32) FApd[id]);
    else
      printf("   A32 Disabled\n");
    
    printf("   Multiblock VME Address Range 0x%08x - 0x%08x\n",ambMin,ambMax);
  }else{
    printf(" Alternate VME Addressing: Multiblock Disabled\n");
    if(addr32&FA_A32_ENABLE)
      printf("   A32 Enabled at VME base 0x%08x or CPU addr 0x%08x\n",a32Base,(UINT32) FApd[id]);
    else
      printf("   A32 Disabled\n");
  }

  if(ctrl1&FA_INT_ENABLE_MASK) {
    printf("\n  Interrupts ENABLED: ");
    if(ctrl1&FA_ENABLE_BLKLVL_INT) printf(" on Block Level(%d)",blevel);
    if(moller)
      if(ctrl1&FA_ENABLE_ERROR_INT)  printf(" Front panel pulse (Moller Data) ");
    else
      if(ctrl1&FA_ENABLE_ERROR_INT)  printf(" on Error Condition ");

    printf("\n");
    printf("  VME INT Vector = 0x%x  Level = %d\n",(intr&FA_INT_VEC_MASK),((intr&FA_INT_LEVEL_MASK)>>8));
  }

  printf("\n Signal Sources: \n");

  if((ctrl1&FA_REF_CLK_MASK)==FA_REF_CLK_INTERNAL) {
    printf("   Ref Clock : Internal\n");
  }else if((ctrl1&FA_REF_CLK_MASK)==FA_REF_CLK_P0) {
    printf("   Ref Clock : VXS\n");
  }else if((ctrl1&FA_REF_CLK_MASK)==FA_REF_CLK_FP) {
    printf("   Ref Clock : Front Panel\n");
  }else if((ctrl1&FA_REF_CLK_MASK)==FA_REF_CLK_P2) {
    printf("   Ref Clock : P2 Backplane\n");
  }else{
    printf("   Ref Clock : %d (Undefined)\n",(ctrl1&FA_REF_CLK_MASK));
  }
  switch(ctrl1&FA_TRIG_MASK) {
  case FA_TRIG_INTERNAL:
    printf("   Trig Src  : Internal\n");
    break;
  case FA_TRIG_VME:
    printf("   Trig Src  : VME (Software)\n");
    break;
  case FA_TRIG_P2_ISYNC:
    printf("   Trig Src  : P2 (Async)\n");
    break;
  case FA_TRIG_P2:
    printf("   Trig Src  : P2 (Sync)\n");
    break;
  case FA_TRIG_P0_ISYNC:
    printf("   Trig Src  : VXS (Async)\n");
    break;
  case FA_TRIG_P0:
    printf("   Trig Src  : VXS (Sync)\n");
    break;
  case FA_TRIG_FP_ISYNC:
    printf("   Trig Src  : Front Panel (Async)\n");
    break;
  case FA_TRIG_FP:
    printf("   Trig Src  : Front Panel (Sync)\n");
  }  
  switch(ctrl1&FA_SRESET_MASK) {
  case FA_SRESET_VME:
    printf("   Sync Reset: VME (Software)\n");
    break;
  case FA_SRESET_P2_ISYNC:
    printf("   Sync Reset: P2 (Async)\n");
    break;
  case FA_SRESET_P2:
    printf("   Sync Reset: P2 (Sync)\n");
    break;
  case FA_SRESET_P0_ISYNC:
    printf("   Sync Reset: VXS (Async)\n");
    break;
  case FA_SRESET_P0:
    printf("   Sync Reset: VXS (Sync)\n");
    break;
  case FA_SRESET_FP_ISYNC:
    printf("   Sync Reset: Front Panel (Async)\n");
    break;
  case FA_SRESET_FP:
    printf("   Sync Reset: Front Panel (Sync)\n");
  }  

  if(fadcUseSDC) {
    printf("   SDC       : In Use\n");
  }


  printf("\n Configuration: \n");

  if(ctrl1&FA_ENABLE_INTERNAL_CLK)
    printf("   Internal Clock ON\n");
  else
    printf("   Internal Clock OFF\n");

  if(ctrl1&FA_ENABLE_BERR)
    printf("   Bus Error ENABLED\n");
  else
    printf("   Bus Error DISABLED\n");


  if(ctrl1&FA_ENABLE_MULTIBLOCK) {
    int tP0, tP2;
    tP0 = ctrl1&FA_MB_TOKEN_VIA_P0;
    tP2 = ctrl1&FA_MB_TOKEN_VIA_P2;
    if(tP0) {
      if(ctrl1&FA_FIRST_BOARD)
	printf("   MultiBlock transfer ENABLED (First Board - token via VXS)\n");
      else if(ctrl1&FA_LAST_BOARD)
	printf("   MultiBlock transfer ENABLED (Last Board  - token via VXS)\n");
      else
	printf("   MultiBlock transfer ENABLED (Token via VXS)\n");
    }else if(tP2){
      if(ctrl1&FA_FIRST_BOARD)
	printf("   MultiBlock transfer ENABLED (First Board - token via P2)\n");
      else if(ctrl1&FA_LAST_BOARD)
	printf("   MultiBlock transfer ENABLED (Last Board  - token via P2)\n");
      else
	printf("   MultiBlock transfer ENABLED (Token via P2)\n");
    }else{
      printf("   MultiBlock transfer ENABLED (**NO Tokens enabled**)\n");
    }
  } else {
    printf("   MultiBlock transfer DISABLED\n");
  }

  if(ctrl1&FA_ENABLE_SOFT_TRIG)
    printf("   Software Triggers   ENABLED\n");
  if(ctrl1&FA_ENABLE_SOFT_SRESET)
    printf("   Software Sync Reset ENABLED\n");



  if(moller == 2) {
    printf("\n ADC Processing Configuration: Cannot Access - Moller DAQ active!!\n");
  }else{
    printf("\n ADC Processing Configuration: \n");
    printf("   Channel Disable Mask = 0x%04x\n",adcChanDisabled);
    if(adc_10bit)
      printf("   Resolution           = 10 bits\n");
    else
      printf("   Resolution           = 12 bits\n");
    if(adc_enabled)
      printf("   Mode = %d  (ENABLED)\n",adc_option);
    else
      printf("   Mode = %d  (Disabled)\n",adc_option);
    printf("   Lookback (PL)    = %d ns   Time Window (PTW) = %d ns\n",PL[0],PTW[0]);
    printf("   Time Before Peak = %d ns   Time After Peak   = %d ns\n",NSB[0],NSA[0]);
    printf("   Max Peak Count   = %d \n",NP);
  }


  printf("\n");
  if(csr&FA_CSR_ERROR_MASK) {
    printf("  CSR       Register = 0x%08x - **Error Condition**\n",csr);
  }else if ((csr&FA_CSR_PLL_MASK) != FA_CSR_PLL_MASK) {
    printf("  CSR       Register = 0x%08x - **Clock NOT Present**\n",csr);
  }else {
    printf("  CSR       Register = 0x%08x\n",csr);
  }

  printf("  Control 1 Register = 0x%08x \n",ctrl1);


  if(moller) {
    if((ctrl2&FA_MOLLER_CTRL_ENABLE_MASK)==FA_MOLLER_CTRL_ENABLED) {
      printf("  Control 2 Register = 0x%08x - Enabled for Moller triggers\n",ctrl2);
    }else{
      printf("  Control 2 Register = 0x%08x - Disabled\n",ctrl2);
    }

  }else{
    if((ctrl2&FA_CTRL_ENABLE_MASK)==FA_CTRL_ENABLED) {
      printf("  Control 2 Register = 0x%08x - Enabled for triggers\n",ctrl2);
    }else{
      printf("  Control 2 Register = 0x%08x - Disabled\n",ctrl2);
    }
  }

  printf("  Internal Triggers (Live) = %d\n",itrigCnt);
  printf("  Trigger   Scaler         = %d\n",trigCnt);


  if(csr&FA_CSR_BLOCK_READY) {
    printf("  Blocks in FIFO     = %d  (Block level = %d) - Block Available\n",bcount,blevel);
    printf("  RAM Level (Bytes)  = %d \n",(ramWords*8)); 
  }else if (csr&FA_CSR_EVENT_AVAILABLE) {
    printf("  Events in FIFO     = %d  (Block level = %d) - Data Available\n",count,blevel);
    printf("  RAM Level (Bytes)  = %d \n",(ramWords*8)); 
  }else{
    printf("  Events in FIFO     = %d  (Block level = %d)\n",count,blevel);
  }

}

void 
faGStatus(int sflag)
{
  int ii;

  for (ii=0;ii<nfadc;ii++) {
    faStatus(fadcID[ii],sflag);
  }

}



/***********************
 *
 *  faSetProcMode - Setup ADC processing modes.
 *
 *
 */
int
faSetProcMode(int id, int pmode, unsigned int PL, unsigned int PTW, 
	      unsigned int NSB, unsigned int NSA, unsigned int NP, int bank)
{
  
  int err=0;
  unsigned int ptw_last_adr, ptw_max_buf;


  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetProcMode: ERROR : FADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if((pmode<=0)||(pmode>8)) {
    printf("faSetProcMode: ERROR: Processing mode (%d) out of range (pmode= 1-8)\n",pmode);
    return(ERROR);
  }else{
    if((pmode>3)&&(pmode<8)) {
      printf("faSetProcMode: ERROR: Processing mode (%d) not implemented \n",pmode);
    }
  }

  if(NP>4) {
    printf("faSetProcMode: ERROR: Invalid Peak count %d (must be 0-4)\n",NP);
    return(ERROR);
  }

  /*Defaults */
  if((PL==0)||(PL>FA_ADC_MAX_PL))  PL  = FA_ADC_DEFAULT_PL;
  if((PTW==0)||(PTW>FA_ADC_MAX_PTW)) PTW = FA_ADC_DEFAULT_PTW;
  if((NSB==0)||(NSB>FA_ADC_MAX_NSB)) NSB = FA_ADC_DEFAULT_NSB;
  if((NSA==0)||(NSA>FA_ADC_MAX_NSA)) NSA = FA_ADC_DEFAULT_NSA;
  if((NP==0)&&(pmode!=FA_ADC_PROC_MODE_WINDOW))  NP = FA_ADC_DEFAULT_NP;

  /* Consistancy check */
  if(PTW > PL) {
    err++;
    printf("faSetProcMode: ERROR: Window must be <= Latency\n"); 
  }
  if(((NSB+NSA)%2)==0) {
    err++;
    printf("faSetProcMode: ERROR: NSB+NSA must be an odd number\n"); 
  }

  /* Calculate Proc parameters */
  ptw_max_buf  = (unsigned int) (2016/(PTW + 8));
  ptw_last_adr = ptw_max_buf * (PTW + 8) - 1;

  if(bank==1) {   /* Program channels 1-8 */
    FAp[id]->adc_cfg[0] = (pmode-1);
    FAp[id]->adc_pl[0] = PL;
    FAp[id]->adc_ptw[0] = PTW;
    FAp[id]->adc_nsb[0] = NSB;
    FAp[id]->adc_nsa[0] = NSA;
    FAp[id]->adc_cfg[0] = ((pmode-1) | (NP<<4) | FA_ADC_PROC_ENABLE | ((fadcChanDisable[id]&0xff)<<8));
    FAp[id]->ptw_max_buf[0]  = ptw_max_buf;
    FAp[id]->ptw_last_adr[0] = ptw_last_adr;
  }else if(bank==2) {  /*Program channels 9-16 */
    FAp[id]->adc_cfg[1] = (pmode-1);
    FAp[id]->adc_pl[1] = PL;
    FAp[id]->adc_ptw[1] = PTW;
    FAp[id]->adc_nsb[1] = NSB;
    FAp[id]->adc_nsa[1] = NSA;
    FAp[id]->adc_cfg[1] = ((pmode-1) | (NP<<4) | FA_ADC_PROC_ENABLE | (fadcChanDisable[id]&0xff00));
    FAp[id]->ptw_max_buf[1]  = ptw_max_buf;
    FAp[id]->ptw_last_adr[1] = ptw_last_adr;
  }else{   /* Program All channels */
    FAp[id]->adc_cfg[0] = (pmode-1);
    FAp[id]->adc_cfg[1] = (pmode-1);
    FAp[id]->adc_pl[0] = PL;
    FAp[id]->adc_pl[1] = PL;
    FAp[id]->adc_ptw[0] = PTW;
    FAp[id]->adc_ptw[1] = PTW;
    FAp[id]->adc_nsb[0] = NSB;
    FAp[id]->adc_nsb[1] = NSB;
    FAp[id]->adc_nsa[0] = NSA;
    FAp[id]->adc_nsa[1] = NSA;
    FAp[id]->adc_cfg[0] = ((pmode-1) | (NP<<4) | FA_ADC_PROC_ENABLE | ((fadcChanDisable[id]&0xff)<<8));
    FAp[id]->adc_cfg[1] = ((pmode-1) | (NP<<4) | FA_ADC_PROC_ENABLE | (fadcChanDisable[id]&0xff00));
    FAp[id]->ptw_max_buf[0]  = ptw_max_buf;
    FAp[id]->ptw_last_adr[0] = ptw_last_adr;
    FAp[id]->ptw_max_buf[1]  = ptw_max_buf;
    FAp[id]->ptw_last_adr[1] = ptw_last_adr;
  }

  return(OK);
}

void
faGSetProcMode(int pmode, unsigned int PL, unsigned int PTW, 
	      unsigned int NSB, unsigned int NSA, unsigned int NP, int bank)
{
  int ii, res;

  for (ii=0;ii<nfadc;ii++) {
    res = faSetProcMode(fadcID[ii],pmode,PL,PTW,NSB,NSA,NP,bank);
    if(res<0) printf("ERROR: slot %d, in faSetProcMode()\n",fadcID[ii]);
  }
}


/*************************************************************************************
 *
 *  faItrigBurstConfig - Setup Internal Trigger Burst control Parameters
 * 
 *   ntrig        = max triggers (1-128) allowed in Burst Window 
 *   burst_window = size (in clock ticks 4ns/tick) of Burst window (1 - 4 microsec)
 *   busy_period  = size (in clocks) of busy period to wait after max triggers reached
 *                   (0 - 262 microsec)
 */
int
faItrigBurstConfig(int id, unsigned int ntrig, 
		   unsigned int burst_window, unsigned int busy_period)
{


  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faItrigBurstConfig: ERROR : FADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  /* Set Defaults */
  if((ntrig==0)||(ntrig>128))                    ntrig        = 4;
  if((burst_window<0x100)||(burst_window>0x3ff)) burst_window = 0x200;
  if((busy_period==0)||(busy_period>0xffff))     busy_period  = 0x800;

  FAp[id]->itrig_burst_count = ntrig;
  FAp[id]->itrig_burst_ctrl  = ((busy_period)<<16) | burst_window;
 
  return(OK);
}
/*
 * Set Internal trigger pulse width and deadtime between triggers 
 *   Range for each :   4ns <-> 1020ns
 *
 *    Units are in clock ticks (4ns/tick)
 */
unsigned int
faItrigControl(int id, unsigned short itrig_width, unsigned short itrig_dt)
{
  unsigned int retval=0;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faItrigControl: ERROR : FADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(0xffffffff);
  }

  /* If both parameters = 0 then just return the current value */
  if((itrig_width==0)&&(itrig_dt==0)) {
    retval = FAp[id]->itrig_cfg;
  }else{
    if((itrig_width==0)||(itrig_width>255))    itrig_width = 0xc; /* default 48ns */
    if((itrig_dt==0)||(itrig_dt>255))          itrig_dt    = 0xa; /* default 40ns */

    FAp[id]->itrig_cfg = (itrig_width<<16)|itrig_dt;
    retval = FAp[id]->itrig_cfg;
  }
 
  return(retval);
}



/**************************************************************************************
 *
 *  faReadBlock - General Data readout routine
 *
 *    id    - Slot number of module to read
 *    data  - local memory address to place data
 *    nwrds - Max number of words to transfer
 *    rflag - Readout Flag
 *              0 - programmed I/O from the specified board
 *              1 - DMA transfer using Universe/Tempe DMA Engine 
 *                    (DMA VME transfer Mode must be setup prior)
 *              2 - Multiblock DMA transfer (Multiblock must be enabled
 *                     and daisychain in place or BDC being used)
 */
int
faReadBlock(int id, volatile UINT32 *data, int nwrds, int rflag)
{
  int ii, blknum, evnum1, evnum2, chan;
  int stat, retVal, xferCount;
  int dCnt, berr=0;
  int dummy=0;
  volatile unsigned int *laddr;
  unsigned int status, bhead, btail, ehead, val;
  unsigned int vmeAdr, csr;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faReadEvent: ERROR : FADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if(data==NULL) {
    logMsg("faReadEvent: ERROR: Invalid Destination address\n",0,0,0,0,0,0);
    return(ERROR);
  }

  if(nwrds <= 0) nwrds= (FA_MAX_ADC_CHANNELS*FA_MAX_DATA_PER_CHANNEL) + 8;

  if(rflag >= 1) { /* Block Transfers */
    
    /*Assume that the DMA programming is already setup. */
    /* Don't Bother checking if there is valid data - that should be done prior
       to calling the read routine */

    /* Check for 8 byte boundary for address - insert dummy word (Slot 0 FADC Dummy DATA)*/
    if((unsigned long) (data)&0x7) {
      *data = FA_DUMMY_DATA;
      dummy = 1;
      laddr = (data + 1);
    } else {
      dummy = 0;
      laddr = data;
    }

    if(rflag == 2) { /* Multiblock Mode */
      if(((FAp[id]->ctrl1)&FA_FIRST_BOARD)==0) {
	logMsg("faReadEvent: ERROR: FADC in slot %d is not First Board\n",id,0,0,0,0,0);
	return(ERROR);
      }
      vmeAdr = (unsigned int)(FApmb) - fadcA32Offset;
    }else{
      vmeAdr = (unsigned int)(FApd[id]) - fadcA32Offset;
    }
    retVal = sysVmeDmaSend((UINT32)laddr, vmeAdr, (nwrds<<2), 0);
    if(retVal |= 0) {
      logMsg("faReadEvent: ERROR in DMA transfer Initialization 0x%x\n",retVal,0,0,0,0,0);
      return(retVal);
    }
    /* Wait until Done or Error */
    retVal = sysVmeDmaDone(10000,1);

    if(retVal > 0) {
      /* Check to see that Bus error was generated by FADC */
      if(rflag == 2) {
	csr = FAp[fadcMaxSlot]->csr;  /* from Last FADC */
	stat = (csr)&FA_CSR_BERR_STATUS;  /* from Last FADC */
      }else{
	stat = (FAp[id]->csr)&FA_CSR_BERR_STATUS;  /* from FADC id */
      }
      if((retVal>0) && (stat)) {
	  xferCount = (nwrds - (retVal>>2) + dummy);  /* Number of Longwords transfered */
        return(xferCount); /* Return number of data words transfered */
      }else{
	xferCount = (nwrds - (retVal>>2) + dummy);  /* Number of Longwords transfered */
	logMsg("faReadEvent: DMA transfer terminated by unknown BUS Error (csr=0x%x nwrds=%d)\n",csr,xferCount,0,0,0,0);
	return(ERROR);
      }
    } else if (retVal == 0){ /* Block Error finished without Bus Error */
      logMsg("faReadEvent: WARN: DMA transfer terminated by word count 0x%x\n",nwrds,0,0,0,0,0);
      return(nwrds);
    } else {  /* Error in DMA */
      logMsg("faReadEvent: ERROR: sysVmeDmaDone returned an Error\n",0,0,0,0,0,0);
      return(retVal);
    }

  } else {  /*Programmed IO */

   /* Check if Bus Errors are enabled. If so then disable for Prog I/O reading */
    berr = (FAp[id]->ctrl1)&FA_ENABLE_BERR;
    if(berr)
      FAp[id]->ctrl1 &= ~FA_ENABLE_BERR;

    dCnt = 0;
    /* Read Block Header - should be first word */
    bhead = (unsigned int) *FApd[id];
    if((bhead&FA_DATA_TYPE_DEFINE)&&((bhead&FA_DATA_TYPE_MASK) == FA_DATA_BLOCK_HEADER)) {
      blknum = bhead&FA_DATA_BLKNUM_MASK;
      ehead = (unsigned int) *FApd[id];
      evnum1 = ehead&FA_DATA_TRIGNUM_MASK;
      data[dCnt] = bhead;
      dCnt++;
      data[dCnt] = ehead;
      dCnt++;
    }else{
      /* We got bad data - Check if there is any data at all */
      if(((FAp[id]->ev_count)&FA_EVENT_COUNT_MASK)==0) {
	logMsg("faReadEvent: FIFO Empty (0x%08x)\n",bhead,0,0,0,0,0);
	return(0);
      } else {
	logMsg("faReadEvent: ERROR: Invalid Header Word 0x%08x\n",bhead,0,0,0,0,0);
	return(ERROR);
      }
    }

    ii=0;
    while(ii<nwrds) {
      data[ii+2] = (unsigned int) *FApd[id];
      if((data[ii+2]&FA_DATA_TYPE_DEFINE)&&((data[ii+2]&FA_DATA_TYPE_MASK) == FA_DATA_BLOCK_TRAILER))
	break;
      ii++;
    }
    ii++;
    dCnt += ii;


    if(berr)
      FAp[id]->ctrl1 |= FA_ENABLE_BERR;

    return(dCnt);
  }

  return(OK);
}


int
faPrintBlock(int id, int rflag)
{

  int ii, blknum, evnum1, evnum2, chan;
  int nwrds=32768, dCnt, berr=0;
  unsigned int data, status, bhead, btail, ehead, val;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    printf("faPrintEvent: ERROR : FADC in slot %d is not initialized \n");
    return(ERROR);
  }

  /* Check if data available */
  if(((FAp[id]->ev_count)&FA_EVENT_COUNT_MASK)==0) {
    printf("faPrintEvent: ERROR: FIFO Empty\n");
    return(0);
  }

  /* Check if Bus Errors are enabled. If so then disable for reading */
  berr = (FAp[id]->ctrl1)&FA_ENABLE_BERR;
  if(berr)
    FAp[id]->ctrl1 &= ~FA_ENABLE_BERR;
  
  dCnt = 0;
  /* Read Block Header - should be first word */
  bhead = (unsigned int) *FApd[id];
  if((bhead&FA_DATA_TYPE_DEFINE)&&((bhead&FA_DATA_TYPE_MASK) == FA_DATA_BLOCK_HEADER)) {
    blknum = bhead&FA_DATA_BLKNUM_MASK;
    ehead = (unsigned int) *FApd[id];
    evnum1 = ehead&FA_DATA_TRIGNUM_MASK;
    faDataDecode(bhead);
    dCnt++;
    faDataDecode(ehead);
    dCnt++;
  }else{
    /* We got bad data - Check if there is any data at all */
    if(((FAp[id]->ev_count)&FA_EVENT_COUNT_MASK)==0) {
      logMsg("faReadEvent: FIFO Empty (0x%08x)\n",bhead,0,0,0,0,0);
      return(0);
    } else {
      logMsg("faReadEvent: ERROR: Invalid Header Word 0x%08x\n",bhead,0,0,0,0,0);
      return(ERROR);
    }
  }
  
  ii=0;
  while(ii<nwrds) {
    data = (unsigned int) *FApd[id];
    faDataDecode(data);
    if((data&FA_DATA_TYPE_DEFINE)&&((data&FA_DATA_TYPE_MASK) == FA_DATA_BLOCK_TRAILER))
      break;
    if((data&FA_DATA_TYPE_DEFINE)&&((data&FA_DATA_TYPE_MASK) == FA_DATA_INVALID))
      break;
    ii++;
  }
  ii++;
  dCnt += ii;


  if(berr)
    FAp[id]->ctrl1 |= FA_ENABLE_BERR;
  
  return(dCnt);
  
}




/*****************************************************************************/

unsigned int
faReadCSR(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faReadCSR: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(0);
  }
  
  return(FAp[id]->csr);
}


void
faClear(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faClear: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->csr = FA_CSR_SOFT_RESET;

}
void
faGClear()
{

  int ii, id;

  for(ii=0;ii<nfadc;ii++) {
    id = fadcID[ii];
    if((id<=0) || (id>21) || (FAp[id] == NULL)) {
      logMsg("faGClear: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    }else{
      FAp[id]->csr = FA_CSR_SOFT_RESET;
    }
  }

}

void
faClearError(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faClearErr: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->csr = FA_CSR_ERROR_CLEAR;

}
void
faGClearError()
{

  int ii, id;

  for(ii=0;ii<nfadc;ii++) {
    id = fadcID[ii];
    if((id<=0) || (id>21) || (FAp[id] == NULL)) {
      logMsg("faGClearErr: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    }else{
      FAp[id]->csr = FA_CSR_ERROR_CLEAR;
    }
  }

}


void
faReset(int id, int iFlag)
{
  unsigned int a32addr, addrMB;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faReset: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  a32addr = FAp[id]->adr32;
  addrMB  = FAp[id]->adr_mb;

  FAp[id]->csr = FA_CSR_HARD_RESET;
  taskDelay(2);

  FAp[id]->adr32  = a32addr;
  FAp[id]->adr_mb = addrMB ;

}

void
faResetToken(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faResetToken: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->reset = FA_RESET_TOKEN;
}

void
faSetCalib(int id, unsigned short sdelay, unsigned short tdelay)
{
  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetCalib: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->delay = (sdelay<<16) | tdelay;

}

void
faChanDisable(int id, unsigned short cmask)
{
  unsigned short temp[2];

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faChanDisable: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  fadcChanDisable[id] = cmask;  /* Set Global Variable */

  /* Read Other Config Info */
  temp[0] = FAp[id]->adc_cfg[0]&0xff;
  temp[1] = FAp[id]->adc_cfg[1]&0xff;

  /* Write New Disable Mask */
  FAp[id]->adc_cfg[0]  = (cmask<<8)     | temp[0];
  FAp[id]->adc_cfg[1]  = (cmask&0xff00) | temp[1];

}


void
faEnable(int id, int eflag, int bank)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faEnable: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  if(bank==1) {   /* Program channels 1-8 */
    FAp[id]->ctrl1 |= (FA_ENABLE_CHAN_0_7);
  }else if(bank==2) {  /*Program channels 9-16 */
    FAp[id]->ctrl1 |= (FA_ENABLE_CHAN_8_15);
  }else{   /* Program All channels */
    FAp[id]->ctrl1 |= (FA_ENABLE_CHAN_0_7 | FA_ENABLE_CHAN_8_15);
  }

  if(eflag) {  /* Enable Internal Trigger logic as well*/
    FAp[id]->ctrl2 = FA_CTRL_GO | FA_CTRL_ENABLE_TRIG | FA_CTRL_ENABLE_SRESET | 
                     FA_CTRL_ENABLE_INT_TRIG;
  }else{
    FAp[id]->ctrl2 = FA_CTRL_GO | FA_CTRL_ENABLE_TRIG | FA_CTRL_ENABLE_SRESET;
  }
}
void
faGEnable(int eflag, int bank)
{
  int ii;

  for(ii=0;ii<nfadc;ii++)
    faEnable(fadcID[ii],eflag,bank);

  if(fadcUseSDC)
    faSDC_Enable(1);

}

void
faDisable(int id, int eflag)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faDisable: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  if(eflag)
    FAp[id]->ctrl2 = 0;   /* Turn FIFO Transfer off as well */
  else
    FAp[id]->ctrl2 = FA_CTRL_GO;
}
void
faGDisable(int eflag)
{
  int ii;

  if(fadcUseSDC)
    faSDC_Disable();

  for(ii=0;ii<nfadc;ii++)
    faDisable(fadcID[ii],eflag);

}



void
faTrig(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faTrig: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  if((FAp[id]->ctrl1)&(FA_ENABLE_SOFT_TRIG))
    FAp[id]->csr = FA_CSR_TRIGGER;
  else
    logMsg("faTrig: ERROR: Software Triggers not enabled",0,0,0,0,0,0);
}
void
faGTrig()
{
  int ii;

  for(ii=0;ii<nfadc;ii++)
    faTrig(fadcID[ii]);
}



void
faSync(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSync: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  if((FAp[id]->ctrl1)&(FA_ENABLE_SOFT_SRESET))
    FAp[id]->csr = FA_CSR_SYNC;
  else
    logMsg("faSync: ERROR: Software Sync Resets not enabled",0,0,0,0,0,0);
}



/* Return Event/Block count for ADC in slot id */
int
faDready(int id, int dflag)
{
  unsigned int dcnt=0;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faDready: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if(dflag)
    dcnt = (FAp[id]->blk_count)&FA_BLOCK_COUNT_MASK;
  else
    dcnt = (FAp[id]->ev_count)&FA_EVENT_COUNT_MASK;

  
  return(dcnt);
}

/* Return a Block Ready status for ADC. If Block Level is =1 then return Event Ready status */
int
faBready(int id)
{
  int stat=0;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faBready: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if(fadcBlockLevel==1) {
    stat = (FAp[id]->csr)&FA_CSR_BLOCK_LEVEL_FLAG;
  }else{
    stat = (FAp[id]->csr)&FA_CSR_BLOCK_READY;
  }

  if(stat)
    return(1);
  else
    return(0);
}

unsigned int
faGBready()
{
  int ii, id, stat=0;
  unsigned int dmask=0;
  
  for(ii-0;ii<nfadc;ii++) {
    id = fadcID[ii];

    if(fadcBlockLevel==1) {
      stat = (FAp[id]->csr)&FA_CSR_BLOCK_LEVEL_FLAG;
    }else{
      stat = (FAp[id]->csr)&FA_CSR_BLOCK_READY;
    }
    
    if(stat)
      dmask |= (1<<id);
  }

  return(dmask);
}


/* Return Number of bits of resolution for FADCs */
int
faGetRes(int id)
{
  unsigned short rez[2];

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faGetRes: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }
  
  rez[0] = FA_ADC_BIT_MASK & FAp[id]->adc_status1[0];
  rez[1] = FA_ADC_BIT_MASK & FAp[id]->adc_status1[1];

  if(rez[0] != rez[1])
    return(ERROR);
  
  if(rez[0]==FA_ADC_NBITS_12)
    return(12);
  else
    return(10);
}


/* if val>0 then set the busy level, if val=0 then read it back.
   if bflag>0 then force the module Busy */
int
faBusyLevel(int id, unsigned int val, int bflag)
{
  unsigned int blreg=0;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faBusyLevel: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }
  if(val>FA_BUSY_LEVEL_MASK)
    return(ERROR);

  /* if Val > 0 then set the Level else leave it alone*/
  if(val) {
    if(bflag)
      FAp[id]->busy_level = (val | FA_FORCE_BUSY);
    else
      FAp[id]->busy_level = val;
  }else{
    blreg = FAp[id]->busy_level;
    if(bflag)
      FAp[id]->busy_level = (blreg | FA_FORCE_BUSY);
  }

  return((blreg&FA_BUSY_LEVEL_MASK));
}

int
faBusy(int id)
{
  unsigned int blreg=0;
  unsigned int dreg=0;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faBusy: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  blreg = (FAp[id]->busy_level)&FA_BUSY_LEVEL_MASK;
  dreg  = (FAp[id]->ram_word_count)&FA_RAM_DATA_MASK;

  if(dreg>=blreg)
    return(1);
  else
    return(0);
}


void
faEnableSoftTrig(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faEnableSoftTrig: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  /* Clear the source */
  FAp[id]->ctrl1 &= ~FA_TRIG_MASK;
  /* Set Source and Enable*/
  FAp[id]->ctrl1 |= (FA_TRIG_VME | FA_ENABLE_SOFT_TRIG);
}
void
faGEnableSoftTrig()
{
  int ii, id;

  for(ii=0;ii<nfadc;ii++) {
    id = fadcID[ii];
    faEnableSoftTrig(id);
  }

}


void
faDisableSoftTrig(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faDisableSoftTrig: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->ctrl1 &= ~FA_ENABLE_SOFT_TRIG;

}

void
faEnableSoftSync(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faEnableSoftSync: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  /* Clear the source */
  FAp[id]->ctrl1 &= ~FA_SRESET_MASK;
  /* Set Source and Enable*/
  FAp[id]->ctrl1 |= (FA_SRESET_VME | FA_ENABLE_SOFT_SRESET);
}

void
faDisableSoftSync(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faDisableSoftSync: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->ctrl1 &= ~FA_ENABLE_SOFT_SRESET;

}

void
faEnableClk(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faEnableClk: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->ctrl1 |= (FA_REF_CLK_INTERNAL|FA_ENABLE_INTERNAL_CLK);

}

void
faDisableClk(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faDisableClk: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->ctrl1 &= ~FA_ENABLE_INTERNAL_CLK;

}


void
faEnableBusError(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faEnableBusError: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->ctrl1 |= FA_ENABLE_BERR;

}
void
faGEnableBusError()
{
  int ii;

  for(ii=0;ii<nfadc;ii++)
    FAp[fadcID[ii]]->ctrl1 |= FA_ENABLE_BERR;

}


void
faDisableBusError(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faDisableBusError: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->ctrl1 &= ~FA_ENABLE_BERR;

}


void
faEnableMultiBlock(int tflag)
{
  int ii, id;
  unsigned int mode;

  if((nfadc <= 1) || (FAp[fadcID[0]] == NULL)) {
    logMsg("faEnableMultiBlock: ERROR : Cannot Enable MultiBlock mode \n",0,0,0,0,0,0);
    return;
  }

  /* if token = 0 then send via P2 else via VXS */
  if(tflag)
    mode = (FA_ENABLE_MULTIBLOCK | FA_MB_TOKEN_VIA_P0);
  else
    mode = (FA_ENABLE_MULTIBLOCK | FA_MB_TOKEN_VIA_P2);
    
  for(ii=0;ii<nfadc;ii++) {
    id = fadcID[ii];
    FAp[id]->ctrl1 |= mode;
    faDisableBusError(id);
    if(id == fadcMinSlot) 
      FAp[id]->ctrl1 |= FA_FIRST_BOARD;
    if(id == fadcMaxSlot) {
      FAp[id]->ctrl1 |= FA_LAST_BOARD;
      faEnableBusError(id);   /* Enable Bus Error only on Last Board */
    }
  }

}

void
faDisableMultiBlock()
{
  int ii;

  if((nfadc <= 1) || (FAp[fadcID[0]] == NULL)) {
    logMsg("faDisableMultiBlock: ERROR : Cannot Disable MultiBlock Mode\n",0,0,0,0,0,0);
    return;
  }
  
  for(ii=0;ii<nfadc;ii++)
    FAp[fadcID[ii]]->ctrl1 &= ~FA_ENABLE_MULTIBLOCK;

}



int
faSetBlockLevel(int id, int level)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetBlockLevel: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }
  
  if(level<=0) level = 1;
  FAp[id]->blk_level = level;
  fadcBlockLevel = level;

  return((FAp[id]->blk_level)&FA_BLOCK_LEVEL_MASK);

}
void
faGSetBlockLevel(int level)
{
  int ii;

  if(level<=0) level = 1;
  for(ii=0;ii<nfadc;ii++)
    FAp[fadcID[ii]]->blk_level = level;

  fadcBlockLevel = level;
}

int
faSetClkSource(int id, int source)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetClkSource: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }
  
  FAp[id]->ctrl1 &= ~FA_REF_CLK_SEL_MASK;
  if((source<0)||(source>7)) source = FA_REF_CLK_INTERNAL;
  FAp[id]->ctrl1 |= source;

  return((FAp[id]->ctrl1)&FA_REF_CLK_SEL_MASK);

}

int
faSetTrigSource(int id, int source)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetTrigSource: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }
  
  FAp[id]->ctrl1 &= ~FA_TRIG_SEL_MASK;
  if((source<0)||(source>7)) source = FA_TRIG_FP_ISYNC;
  FAp[id]->ctrl1 |= source;

  return((FAp[id]->ctrl1)&FA_TRIG_SEL_MASK);

}

int
faSetSyncSource(int id, int source)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetSyncSource: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }
  
  FAp[id]->ctrl1 &= ~FA_SRESET_SEL_MASK;
  if((source<0)||(source>7)) source = FA_SRESET_FP_ISYNC;
  FAp[id]->ctrl1 |= source;

  return((FAp[id]->ctrl1)&FA_SRESET_SEL_MASK);

}

/* Enable Front Panel Inputs (and Disable software triggers/syncs 
   but leave the clock source alone */
void
faEnableFP(int id)
{

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faEnableFP: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->ctrl1 &= ~(FA_TRIG_SEL_MASK | FA_SRESET_SEL_MASK | FA_ENABLE_SOFT_SRESET | FA_ENABLE_SOFT_TRIG);
  FAp[id]->ctrl1 |= (FA_TRIG_FP_ISYNC | FA_SRESET_FP_ISYNC);

}


int
faSetThreshold(int id, unsigned short tvalue, unsigned short chmask)
{

  int ii;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetThreshold: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if(chmask==0) chmask = 0xffff;  /* Set All channels the same */

  for(ii=0;ii<FA_MAX_ADC_CHANNELS;ii++) {
    if((1<<ii)&chmask) FAp[id]->adc_thres[ii] = tvalue;
  }

  return(OK);
}

int
faPrintThreshold(int id)
{
  int ii;
  unsigned short tval[FA_MAX_ADC_CHANNELS];

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faPrintThreshold: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  for(ii=0;ii<FA_MAX_ADC_CHANNELS;ii++)
    tval[ii] = FAp[id]->adc_thres[ii];


  printf(" Threshold Settings for FADC in slot %d:",id);
  for(ii=0;ii<FA_MAX_ADC_CHANNELS;ii++) {
    if((ii%4)==0) printf("\n");
    printf("Chan %2d: %3d   ",(ii+1),tval[ii]);
  }
  printf("\n");


  return(OK);
}


int
faSetDAC(int id, unsigned short dvalue, unsigned short chmask)
{
  int ii;

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faSetDAC: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if(chmask==0) chmask = 0xffff;  /* Set All channels the same */

  if(dvalue>0xff) {
    logMsg("faSetDAC: ERROR : DAC value (%d) out of range (0-255) \n",
	   dvalue,0,0,0,0,0);
    return(ERROR);
  }

  for(ii=0;ii<FA_MAX_ADC_CHANNELS;ii++)
    if((1<<ii)&chmask) FAp[id]->dac[ii] = (dvalue&FA_DAC_VALUE_MASK) | FA_DAC_LOAD;

  return(OK);
}

void
faPrintDAC(int id)
{
  int ii;
  unsigned short dval[FA_MAX_ADC_CHANNELS];

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faPrintDAC: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  for(ii=0;ii<FA_MAX_ADC_CHANNELS;ii++)
    dval[ii] = (FAp[id]->dac[ii])&FA_DAC_VALUE_MASK;


  printf(" DAC Settings for FADC in slot %d:",id);
  for(ii=0;ii<FA_MAX_ADC_CHANNELS;ii++) {
    if((ii%4)==0) printf("\n");
    printf("Chan %2d: %3d   ",(ii+1),dval[ii]);
  }
  printf("\n");

}



/* -------------------------------------------------------------------------------------

    Utility routines
*/

void 
faPrintAuxScal(int id)
{
  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faPrintAuxScal: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  printf("Auxillary Scalers:     FPGA1          FPGA2\n");
  printf("       Word Count:         %d              %d\n",FAp[id]->aux_scal[0], FAp[id]->aux_scal[1]);
  printf("       Headers   :         %d              %d\n",FAp[id]->aux_scal[2], FAp[id]->aux_scal[3]);
  printf("       Trailers  :         %d              %d\n",FAp[id]->aux_scal[4], FAp[id]->aux_scal[5]);

  return;
}

void 
faPrintFifoStatus(int id)
{
  unsigned int ibuf, bbuf, obuf, dflow;
  unsigned int wc[2],mt[2],full[2];
  unsigned int rdy[2];

  if(id==0) id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("faPrintFifoStatus: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  dflow = FAp[id]->dataflow_status;
  ibuf = (FAp[id]->status[0])&0xdfffdfff;
  bbuf = (FAp[id]->status[1])&0x1fff1fff;
  obuf = (FAp[id]->status[2])&0x3fff3fff;

  printf("Fifo Buffers Status (DataFlow Status = 0x%08x\n",dflow);

  mt[0]  = full[0] = 0;
  wc[0]  = ibuf&0x7ff;
  rdy[0] = (ibuf&0x8000)>>15;
  if(ibuf&0x800) full[0]=1;
  if(ibuf&0x1000) mt[0]=1;

  mt[1]  = full[1] = 0;
  wc[1]  = (ibuf&0x7ff0000)>>16;
  rdy[1] = (ibuf&0x80000000)>>31;
  if(ibuf&0x8000000)  full[1]=1;
  if(ibuf&0x10000000) mt[1]=1;

  printf("  Input Buffer : 0x%08x \n",ibuf);
  printf("    FPGA1: wc=%d   Empty=%d Full=%d Ready=%d\n",wc[0],mt[0],full[0],rdy[0]);
  printf("    FPGA2: wc=%d   Empty=%d Full=%d Ready=%d\n",wc[1],mt[1],full[1],rdy[1]);

  mt[0]=full[0]=0;
  wc[0]   =  bbuf&0x7ff;
  if(bbuf&0x800) full[0]=1;
  if(bbuf&0x1000) mt[0]=1;

  mt[1]=full[1]=0;
  wc[1]   = (bbuf&0x7ff0000)>>16;
  if(bbuf&0x8000000)  full[1]=1;
  if(bbuf&0x10000000) mt[1]=1;

  printf("  Build Buffer : 0x%08x \n",bbuf);
  printf("    FPGA1: wc=%d   Empty=%d Full=%d \n",wc[0],mt[0],full[0]);
  printf("    FPGA2: wc=%d   Empty=%d Full=%d \n",wc[1],mt[1],full[1]);

  mt[0]=full[0]=0;
  wc[0]   =  obuf&0xfff;
  if(obuf&0x1000) full[0]=1;
  if(obuf&0x2000) mt[0]=1;

  mt[1]=full[1]=0;
  wc[1]   = (obuf&0xfff0000)>>16;
  if(obuf&0x10000000)  full[1]=1;
  if(obuf&0x20000000) mt[1]=1;

  printf("  Output Buffer: 0x%08x \n",obuf);
  printf("    FPGA1: wc=%d   Empty=%d Full=%d \n",wc[0],mt[0],full[0]);
  printf("    FPGA2: wc=%d   Empty=%d Full=%d \n",wc[1],mt[1],full[1]);


  return;

}


void 
faDataDecode(unsigned int data)
{
  int i_print = 1;
	static unsigned int type_last = 15;	/* initialize to type FILLER WORD */
	static unsigned int time_last = 0;
	
        if( data & 0x80000000 )		/* data type defining word */
        {
            fadc_data.new_type = 1;
            fadc_data.type = (data & 0x78000000) >> 27;
        }
        else
        {
            fadc_data.new_type = 0;
            fadc_data.type = type_last;
        }
        
	switch( fadc_data.type )
	{
	    case 0:		/* BLOCK HEADER */
	        fadc_data.slot_id_hd = (data & 0x7C00000) >> 22;
	        fadc_data.n_evts = (data & 0x3FF800) >> 11;
	        fadc_data.blk_num = (data & 0x7FF);
	        if( i_print ) 
	            printf("%8X - BLOCK HEADER - slot = %d   n_evts = %d   n_blk = %d\n",
	        	data, fadc_data.slot_id_hd, fadc_data.n_evts, fadc_data.blk_num);
	        break;
	    case 1:		/* BLOCK TRAILER */
	        fadc_data.slot_id_tr = (data & 0x7C00000) >> 22;
	        fadc_data.n_words = (data & 0x3FFFFF);
	        if( i_print ) 
	            printf("%8X - BLOCK TRAILER - slot = %d   n_words = %d\n",
	        	data, fadc_data.slot_id_tr, fadc_data.n_words);
	        break;
	    case 2:		/* EVENT HEADER */
	    	if( fadc_data.new_type )
	    	{
	            fadc_data.evt_num_1 = (data & 0x7FFFFFF);
	            if( i_print ) 
	                printf("%8X - EVENT HEADER 1 - evt_num = %d\n", data, fadc_data.evt_num_1);
	        }    
	    	else
	    	{
	            fadc_data.evt_num_2 = (data & 0x7FFFFFF);
	            if( i_print ) 
	                printf("%8X - EVENT HEADER 2 - evt_num = %d\n", data, fadc_data.evt_num_2);
	    	}
	        break;
	    case 3:		/* TRIGGER TIME */
	    	if( fadc_data.new_type )
	    	{
	            fadc_data.time_1 = (data & 0xFFFFFF);
	            if( i_print ) 
	                printf("%8X - TRIGGER TIME 1 - time = %LX\n", data, fadc_data.time_1);
	            fadc_data.time_now = 1;
	            time_last = 1;
	        }    
	    	else
	    	{
	    	    if( time_last == 1 )
	    	    {
	                fadc_data.time_2 = (data & 0xFFFFFF);
	                if( i_print ) 
	                    printf("%8X - TRIGGER TIME 2 - time = %LX\n", data, fadc_data.time_2);
	                fadc_data.time_now = 2;
	            }    
	    	    else if( time_last == 2 )
	    	    {
	                fadc_data.time_3 = (data & 0xFFFFFF);
	                if( i_print ) 
	                    printf("%8X - TRIGGER TIME 3 - time = %LX\n", data, fadc_data.time_3);
	                fadc_data.time_now = 3;
	            }    
	    	    else if( time_last == 3 )
	    	    {
	                fadc_data.time_4 = (data & 0xFFFFFF);
	                if( i_print ) 
	                    printf("%8X - TRIGGER TIME 4 - time = %LX\n", data, fadc_data.time_4);
	                fadc_data.time_now = 4;
	            }    
	    	    else
	                if( i_print ) 
	                    printf("%8X - TRIGGER TIME - (ERROR)\n", data);
	                
	            time_last = fadc_data.time_now;
	        }    
	        break;
	    case 4:		/* WINDOW RAW DATA */
	    	if( fadc_data.new_type )
	    	{
	            fadc_data.chan = (data & 0x7800000) >> 23;
	            fadc_data.width = (data & 0xFFF);
	            if( i_print ) 
	                printf("%8X - WINDOW RAW DATA - chan = %d   nsamples = %d\n", 
	                	data, fadc_data.chan, fadc_data.width);
	        }    
	    	else
	    	{
	    	    fadc_data.valid_1 = 1;
	    	    fadc_data.valid_2 = 1;
	            fadc_data.adc_1 = (data & 0x1FFF0000) >> 16;
	            if( data & 0x20000000 )
	                fadc_data.valid_1 = 0;
	            fadc_data.adc_2 = (data & 0x1FFF);
	            if( data & 0x2000 )
	                fadc_data.valid_2 = 0;
	            if( i_print ) 
	                printf("%8X - RAW SAMPLES - valid = %d  adc = %d   valid = %d  adc = %d\n", 
	            		 data, fadc_data.valid_1, fadc_data.adc_1, 
	            		 fadc_data.valid_2, fadc_data.adc_2);
	        }    
	        break;
	    case 5:		/* WINDOW SUM */
	    	fadc_data.over = 0; 
	        fadc_data.chan = (data & 0x7800000) >> 23;
	        fadc_data.adc_sum = (data & 0x3FFFFF);
	        if( data & 0x400000 )
	            fadc_data.over = 1;
	        if( i_print ) 
	            printf("%8X - WINDOW SUM - chan = %d   over = %d   adc_sum = %LX\n",
	        		data, fadc_data.chan, fadc_data.over, fadc_data.adc_sum);
	        break;
	    case 6:		/* PULSE RAW DATA */
	    	if( fadc_data.new_type )
	    	{
	            fadc_data.chan = (data & 0x7800000) >> 23;
	            fadc_data.pulse_num = (data & 0x600000) >> 21;
	            fadc_data.thres_bin = (data & 0x3FF);
	            if( i_print ) 
	                printf("%8X - PULSE RAW DATA - chan = %d   pulse # = %d   threshold bin = %d\n", 
	            		 data, fadc_data.chan, fadc_data.pulse_num, fadc_data.thres_bin);
	        }    
	    	else
	    	{
	    	    fadc_data.valid_1 = 1;
	    	    fadc_data.valid_2 = 1;
	            fadc_data.adc_1 = (data & 0x1FFF0000) >> 16;
	            if( data & 0x20000000 )
	                fadc_data.valid_1 = 0;
	            fadc_data.adc_2 = (data & 0x1FFF);
	            if( data & 0x2000 )
	                fadc_data.valid_2 = 0;
	            if( i_print ) 
	                printf("%8X - PULSE RAW SAMPLES - valid = %d  adc = %d   valid = %d  adc = %d\n", 
	            		 data, fadc_data.valid_1, fadc_data.adc_1, 
	            		 fadc_data.valid_2, fadc_data.adc_2);
	        }    
	        break;
	    case 7:		/* PULSE INTEGRAL */
	            fadc_data.chan = (data & 0x7800000) >> 23;
	            fadc_data.pulse_num = (data & 0x600000) >> 21;
	            fadc_data.quality = (data & 0x180000) >> 19;
	            fadc_data.integral = (data & 0x7FFFF);
	            if( i_print ) 
	                printf("%8X - PULSE INTEGRAL - chan = %d   pulse # = %d   quality = %d   integral = %d\n", 
	            		 data, fadc_data.chan, fadc_data.pulse_num, 
	            		 fadc_data.quality, fadc_data.integral);
	        break;
	    case 8:		/* PULSE TIME */
	            fadc_data.chan = (data & 0x7800000) >> 23;
	            fadc_data.pulse_num = (data & 0x600000) >> 21;
	            fadc_data.quality = (data & 0x180000) >> 19;
	            fadc_data.time = (data & 0xFFFF);
	            if( i_print ) 
	                printf("%8X - PULSE TIME - chan = %d   pulse # = %d   quality = %d   time = %d\n", 
	            		 data, fadc_data.chan, fadc_data.pulse_num, 
	            		 fadc_data.quality, fadc_data.time);
	        break;
	    case 9:		/* STREAMING RAW DATA */
	    	if( fadc_data.new_type )
	    	{
	            fadc_data.chan_a = (data & 0x3C00000) >> 22;
	            fadc_data.source_a = (data & 0x4000000) >> 26;
	            fadc_data.chan_b = (data & 0x1E0000) >> 17;
	            fadc_data.source_b = (data & 0x200000) >> 21;
	            if( i_print ) 
	                printf("%8X - STREAMING RAW DATA - ena A = %d  chan A = %d   ena B = %d  chan B = %d\n", 
	            		 data, fadc_data.source_a, fadc_data.chan_a, 
	            		 fadc_data.source_b, fadc_data.chan_b);
	        }    
	    	else
	    	{
	    	    fadc_data.valid_1 = 1;
	    	    fadc_data.valid_2 = 1;
	            fadc_data.adc_1 = (data & 0x1FFF0000) >> 16;
	            if( data & 0x20000000 )
	                fadc_data.valid_1 = 0;
	            fadc_data.adc_2 = (data & 0x1FFF);
	            if( data & 0x2000 )
	                fadc_data.valid_2 = 0;
	    	    fadc_data.group = (data & 0x40000000) >> 30;
	    	    if( fadc_data.group )
	    	    {
	                if( i_print ) 
	                    printf("%8X - RAW SAMPLES B - valid = %d  adc = %d   valid = %d  adc = %d\n", 
	            		 data, fadc_data.valid_1, fadc_data.adc_1, 
	            		 fadc_data.valid_2, fadc_data.adc_2);
	            }		 
	            else
	                if( i_print ) 
	                    printf("%8X - RAW SAMPLES A - valid = %d  adc = %d   valid = %d  adc = %d\n", 
	            		 data, fadc_data.valid_1, fadc_data.adc_1, 
	            		 fadc_data.valid_2, fadc_data.adc_2);	            
	        }    
	        break;
	    case 10:		/* PULSE AMPLITUDE DATA */
	      fadc_data.chan = (data & 0x7800000) >> 23;
	      fadc_data.pulse_num = (data & 0x600000) >> 21;
	      fadc_data.vmin = (data & 0x1FF000) >> 12;
	      fadc_data.vpeak = (data & 0xFFF);
	      if( i_print ) 
		printf("%8X - PULSE V - chan = %d   pulse # = %d   vmin = %d   vpeak = %d\n", 
		       data, fadc_data.chan, fadc_data.pulse_num, 
		       fadc_data.vmin, fadc_data.vpeak);
	        break;

	    case 11:		/* INTERNAL TRIGGER WORD */
	    	fadc_data.trig_type_int = data & 0x7;
	    	fadc_data.trig_state_int = (data & 0x8) >> 3;
	    	fadc_data.evt_num_int = (data & 0xFFF0) >> 4;
	    	fadc_data.err_status_int = (data & 0x10000) >> 16;
	        if( i_print ) 
	            printf("%8X - INTERNAL TRIGGER - type = %d   state = %d   num = %d   error = %d\n",
	        	data, fadc_data.trig_type_int, fadc_data.trig_state_int, fadc_data.evt_num_int,
	        	fadc_data.err_status_int);
	    case 12:		/* UNDEFINED TYPE */
	        if( i_print ) 
	            printf("%8X - UNDEFINED TYPE = %d\n", data, fadc_data.type);
	        break;
	    case 13:		/* END OF EVENT */
	        if( i_print ) 
	            printf("%8X - END OF EVENT = %d\n", data, fadc_data.type);
	        break;
	    case 14:		/* DATA NOT VALID (no data available) */
	        if( i_print ) 
	            printf("%8X - DATA NOT VALID = %d\n", data, fadc_data.type);
	        break;
	    case 15:		/* FILLER WORD */
	        if( i_print ) 
	            printf("%8X - FILLER WORD = %d\n", data, fadc_data.type);
	        break;
	}
	
	type_last = fadc_data.type;	/* save type of current data word */
		   
}        


/***************************************************************************************
   JLAB FADC Signal Distribution Card (SDC) Routines

   cFlag:  controls the configuation of the SDC
          
          0:  Default Mode  Internal CLK, Sync External Trigger and Sync Reset
        > 0:  Pass through mode
                    
   bMask:  mask of Busy enables for the SDC - Do not Enable busy if there is no FADC

*/

int
faSDC_Config(unsigned short cFlag, unsigned short bMask)
{
  int ii;

  if(FASDCp == NULL) {
    logMsg("faSDC_Config: ERROR : Cannot Configure FADC Signal Board \n",0,0,0,0,0,0);
    return(ERROR);
  }

  /* Reset the Board */
  FASDCp->csr = FASDC_CSR_INIT;

  if(cFlag == 0) {
    /* Default - Enable Internal Clock, Sync Trigger and Sync-Reset*/
    FASDCp->ctrl = (FASDC_CTRL_ENABLE_SOFT_TRIG | FASDC_CTRL_ENABLE_SOFT_SRESET) ;
  }else if(cFlag==1) {
    /* Pass Through - */
    FASDCp->ctrl = (FASDC_CTRL_NOSYNC_TRIG | FASDC_CTRL_NOSYNC_SRESET | 
                    FASDC_CTRL_ENABLE_SOFT_TRIG | FASDC_CTRL_ENABLE_SOFT_SRESET);
  }else { 
    /* Moller Level Translator */
    FASDCp->ctrl = (FASDC_CTRL_NOSYNC_TRIG | FASDC_CTRL_NOSYNC_SRESET) ;
  }

  FASDCp->busy_enable = bMask;

  return(OK);
}

void
faSDC_Status(int sFlag)
{

  unsigned short sdc[4];


  if(FASDCp == NULL) {
    printf("faSDC_Status: ERROR : No FADC SDC available \n");
    return;
  }

  sdc[0] = (FASDCp->csr);
  sdc[1] = (FASDCp->ctrl)&FASDC_CTRL_MASK;
  sdc[2] = (FASDCp->busy_enable)&FASDC_BUSY_MASK;
  sdc[3] = (FASDCp->busy_status);


  printf("\nSTATUS for FADC Signal Distribution Card at base address 0x%x \n",(UINT32) FASDCp);
  printf("---------------------------------------------------------------- \n");

  printf(" Board Firmware Rev/ID = 0x%02x\n",((sdc[0]&0xff00)>>8));
  printf(" Registers: \n");
  printf("   CSR         = 0x%04x     Control     = 0x%04x\n",sdc[0],sdc[1]);
  printf("   Busy Enable = 0x%04x     Busy Status = 0x%04x\n",sdc[2],sdc[3]);
  printf("\n");

  if((sdc[1]&FASDC_CTRL_CLK_EXT))
    printf(" Ref Clock : External\n");
  else
    printf(" Ref Clock : Internal\n");

  if((sdc[1]&FASDC_CTRL_ENABLE_SOFT_TRIG)) {
    printf(" Software Triggers\n");
  }else{
    if((sdc[1]&FASDC_CTRL_NOSYNC_TRIG))
      printf(" External Triggers (Pass through)\n");
    else
      printf(" External Triggers (Sync with clock)\n");
  }

  if((sdc[1]&FASDC_CTRL_ENABLE_SOFT_SRESET)) {
    printf(" Software Sync Reset\n");
  }else{
    if((sdc[1]&FASDC_CTRL_NOSYNC_SRESET))
      printf(" External Sync Reset (Pass through)\n");
    else
      printf(" External Sync Reset (Sync with clock)\n");
  }
 
}


void
faSDC_Enable(int nsync)
{

  if(FASDCp == NULL) {
    logMsg("faSDC_Enable: ERROR : No FADC SDC available \n",0,0,0,0,0,0);
    return;
  }
  
  if(nsync != 0) /* FP triggers only */
    FASDCp->ctrl = FASDC_CTRL_ENABLE_SOFT_SRESET;
  else      /* both FP triggers and sync reset */
    FASDCp->ctrl = 0;
}

void
faSDC_Disable()
{

  if(FASDCp == NULL) {
    logMsg("faSDC_Disable: ERROR : No FADC SDC available \n",0,0,0,0,0,0);
    return;
  }
  
  FASDCp->ctrl = (FASDC_CTRL_ENABLE_SOFT_TRIG | FASDC_CTRL_ENABLE_SOFT_SRESET);
}



void
faSDC_Sync()
{

  if(FASDCp == NULL) {
    logMsg("faSDC_Sync: ERROR : No FADC SDC available \n",0,0,0,0,0,0);
    return;
  }
  
  FASDCp->csr = FASDC_CSR_SRESET;
}

void
faSDC_Trig()
{
  if(FASDCp == NULL) {
    logMsg("faSDC_Trig: ERROR : No FADC SDC available \n",0,0,0,0,0,0);
    return;
  }
  
  FASDCp->csr = FASDC_CSR_TRIG;
}

int
faSDC_Busy()
{
  int busy=0;

  if(FASDCp == NULL) {
    logMsg("faSDC_Busy: ERROR : No FADC SDC available \n",0,0,0,0,0,0);
    return;
  }
  
  busy = (FASDCp->csr)&FASDC_CSR_BUSY;

  return(busy);
}
